<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\User; // Import the User model
use App\Models\Order; // Import the Order model
use App\Models\OrderItem; // Import the OrderItem model
use App\Models\Service; // Import the Service model
use App\Models\SchoolService; // Import the SchoolService model
use App\Models\Booking; // Import the Booking model
use App\Models\Event; // Import the Event model
use App\Http\Requests\DoctorStoreRequest; // Import the DoctorStoreRequest
use Illuminate\Support\Facades\Hash; // Import Hash for password hashing
use Illuminate\Support\Str; // Import Str for UUID generation
use App\Notifications\NewOrderNotification; // Import the NewOrderNotification
use Illuminate\Support\Facades\Notification; // Import Notification facade

class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $orders = Order::with('user')->latest()->paginate(10);
        $serviceBookings = Booking::with('user', 'service', 'schoolService')->latest()->paginate(10); // Fetch service and school service bookings
        $eventBookings = Event::with('user')->latest()->paginate(10); // Fetch event bookings (assuming direct user booking)

        return Inertia::render('Admin/Orders/Index', [
            'orders' => $orders,
            'serviceBookings' => $serviceBookings,
            'eventBookings' => $eventBookings,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // For demonstration purposes, create a dummy order and dispatch notification
        $user = User::first(); // Get any user, or a specific admin user

        if (!$user) {
            // Handle case where no user exists
            return redirect()->back()->with('error', 'No user found to create an order.');
        }

        $order = Order::create([
            'id' => Str::uuid(),
            'user_id' => $user->id,
            'total_amount' => 100.00, // Dummy amount
            'status' => 'pending',
            'order_date' => now(),
        ]);

        // Dispatch notification to admin users (e.g., users with role 500)
        $adminUsers = User::where('role', 500)->get();
        Notification::send($adminUsers, new NewOrderNotification($order));

        return redirect()->back()->with('success', 'Dummy order created and notification dispatched.');
    }

    /**
     * Display the specified resource.
     */
    public function show(Order $order)
    {
        $order->load(['user', 'orderItems.item']); // Load user and polymorphic item relationships
        return Inertia::render('Admin/Orders/Show', [
            'order' => $order,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Order $order)
    {
        $validated = $request->validate([
            'status' => ['required', 'string', 'in:pending,completed,cancelled'],
        ]);

        $order->status = $validated['status'];
        $order->save();

        return redirect()->back()->with('success', 'Order status updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Order $order)
    {
        $order->delete();
        return redirect()->back()->with('success', 'Order deleted successfully.');
    }
}
