<?php

namespace App\Http\Controllers;

use App\Models\Portfolio;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class PortfolioController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (request()->wantsJson()) {
            $portfolios = Portfolio::orderBy('order')->get();
            return response()->json($portfolios);
        }
        return Inertia::render('Admin/Portfolios/Index', [
            'portfolios' => Portfolio::orderBy('order')->get(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Portfolios/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'link' => 'nullable|string|max:255',
            'order' => 'required|integer',
        ]);


        $imagePath = $request->file('image')->store('portfolios', 'public');
        $imageUrl = Storage::disk('public')->url($imagePath);

        $portfolio = Portfolio::create([
            'title' => $request->title,
            'description' => $request->description,
            'image_path' => $imagePath,
            'image_url' => $imageUrl,
            'link' => $request->link,
            'order' => $request->order,
        ]);

        if (request()->wantsJson()) {
            return response()->json($portfolio, 201);
        }
        return redirect()->route('admin.portfolios.index')->with('success', 'Portfolio created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $portfolio = Portfolio::findOrFail($id);
        return response()->json($portfolio);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Portfolio $portfolio)
    {
        return Inertia::render('Admin/Portfolios/Edit', [
            'portfolio' => $portfolio,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Portfolio $portfolio)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'link' => 'nullable|string|max:255',
            'order' => 'required|integer',
        ]);

        if ($request->hasFile('image')) {
            Storage::disk('public')->delete($portfolio->image_path);
            $imagePath = $request->file('image')->store('portfolios', 'public');
            $imageUrl = Storage::disk('public')->url($imagePath);
            $portfolio->image_path = $imagePath;
            $portfolio->image_url = $imageUrl;
        }

        $portfolio->update([
            'title' => $request->title,
            'description' => $request->description,
            'link' => $request->link,
            'order' => $request->order,
        ]);

        if (request()->wantsJson()) {
            return response()->json($portfolio);
        }
        return redirect()->route('admin.portfolios.index')->with('success', 'Portfolio updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Portfolio $portfolio)
    {
        Storage::disk('public')->delete($portfolio->image_path);
        $portfolio->delete();

        if (request()->wantsJson()) {
            return response()->json(null, 204);
        }
        return redirect()->route('admin.portfolios.index')->with('success', 'Portfolio deleted successfully.');
    }
}
