<?php

namespace App\Http\Controllers;

use App\Models\Slider;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Inertia\Inertia;

class SliderController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        if (request()->wantsJson()) {
            $sliders = Slider::orderBy('order')->get();
            return response()->json($sliders);
        }
        return Inertia::render('Admin/Sliders/Index', [
            'sliders' => Slider::orderBy('order')->get(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return Inertia::render('Admin/Sliders/Create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'title' => 'required|string|max:255',
                'subtitle' => 'nullable|string|max:255',
                'image' => 'required|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
                'link' => 'nullable|string|max:255',
                'order' => 'required|integer',
            ]);

            $imagePath = "";
            $imageUrl = "";

            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('sliders', 'public');
                $imageUrl= url($imagePath);
            }

            $slider = Slider::create([
                'title' => $request->title,
                'subtitle' => $request->subtitle,
                'image_path' => $imagePath,
                'image_url' => $imageUrl,
                'link' => $request->link,
                'order' => $request->order,
            ]);

            if (request()->wantsJson()) {
                return response()->json($slider, 201);
            }
            return redirect()->route('admin.sliders.index')->with('success', 'Slider created successfully.');
        } catch (\Throwable $th) {
            dd($th);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $slider = Slider::findOrFail($id);
        return response()->json($slider);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Slider $slider)
    {
        return Inertia::render('Admin/Sliders/Edit', [
            'slider' => $slider,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Slider $slider)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'subtitle' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
            'link' => 'nullable|string|max:255',
            'order' => 'required|integer',
        ]);

        if ($request->hasFile('image')) {
            Storage::disk('public')->delete($slider->image_path);
            $imagePath = $request->file('image')->store('sliders', 'public');
            $imageUrl = Storage::disk('public')->url($imagePath);
            $slider->image_path = $imagePath;
            $slider->image_url = $imageUrl;
        }

        $slider->update([
            'title' => $request->title,
            'subtitle' => $request->subtitle,
            'link' => $request->link,
            'order' => $request->order,
        ]);

        if (request()->wantsJson()) {
            return response()->json($slider);
        }
        return redirect()->route('admin.sliders.index')->with('success', 'Slider updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Slider $slider)
    {
        Storage::disk('public')->delete($slider->image_path);
        $slider->delete();

        if (request()->wantsJson()) {
            return response()->json(null, 204);
        }
        return redirect()->route('admin.sliders.index')->with('success', 'Slider deleted successfully.');
    }
}
