<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use App\Models\Booking; // Use Booking model
use App\Models\Order; // Assuming your Order model for product purchases
use App\Models\Service; // Import Service model

use Illuminate\Support\Facades\Auth;

class UserBookingController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        // Fetch bookings for services
        $bookings = Booking::where('user_id', $user->id)
                            ->where('bookable_type', Service::class)
                            ->with('bookable') // Eager load the polymorphic relationship (Service model)
                            ->latest()
                            ->paginate(10);

        $bookings->getCollection()->transform(function ($booking) {
            return [
                'id' => $booking->id,
                'service_name' => $booking->bookable->title, // Access title from the eager-loaded Service model
                'booking_date' => $booking->booking_date,
                'booking_time' => $booking->booking_time,
                'status' => $booking->status,
            ];
        });

        return Inertia::render('User/Bookings/Index', [
            'bookings' => $bookings,
        ]);
    }

    public function show(Booking $booking)
    {
        $this->authorize('view', $booking);

        // Ensure the booking is for a Service
        if ($booking->bookable_type !== Service::class) {
            abort(404, 'Booking not found or is not a service booking.');
        }

        $booking->load('bookable'); // Eager load the polymorphic relationship (Service model)

        return response()->json([
            'booking' => [
                'id' => $booking->id,
                'service_name' => $booking->bookable->title,
                'booking_date' => $booking->booking_date,
                'booking_time' => $booking->booking_time,
                'status' => $booking->status,
                'notes' => $booking->notes,
                'service' => $booking->bookable,
            ]
        ]);
    }

    public function confirm(Booking $booking)
    {
        $this->authorize('update', $booking);

        $booking->update(['status' => 'confirmed']);

        return redirect()->route('user.dashboard')->with('success', 'Booking confirmed successfully.');
    }

    public function reschedule(Booking $booking)
    {
        $this->authorize('update', $booking);

        // For now, we'll just mark it as rescheduled.
        // A more complex implementation would involve a form to select a new date/time.
        $booking->update(['status' => 'rescheduled']);

        return redirect()->route('user.dashboard')->with('success', 'Booking reschedule request sent.');
    }

    public function cancel(Booking $booking)
    {
        $this->authorize('update', $booking);

        $booking->update(['status' => 'cancelled']);

        return redirect()->route('user.dashboard')->with('success', 'Booking cancelled successfully.');
    }
}
