<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\SchoolService; // Import SchoolService model
use App\Models\Booking; // Import Booking model
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;

class UserSchoolServiceBookingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $user = Auth::user();

        $schoolServiceBookings = Booking::where('user_id', $user->id)
                                    ->where('bookable_type', SchoolService::class)
                                    ->with('bookable') // Eager load the polymorphic relationship (SchoolService model)
                                    ->latest()
                                    ->paginate(10);

        $schoolServiceBookings->getCollection()->transform(function ($booking) {
            $totalPrice = (float) $booking->price;
            return [
                'id' => $booking->id,
                'school_service_name' => $booking->bookable->name, // Access name from the eager-loaded SchoolService model
                'booking_date' => $booking->booking_date,
                'quantity' => $booking->quantity,
                'total_price' => $totalPrice,
                'status' => $booking->status,
            ];
        });

        return Inertia::render('User/SchoolServices/Index', [
            'schoolServiceBookings' => $schoolServiceBookings,
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(Booking $booking)
    {
        $this->authorize('view', $booking);

        // Ensure the booking is for a SchoolService
        if ($booking->bookable_type !== SchoolService::class) {
            abort(404, 'Booking not found or is not a school service booking.');
        }

        $booking->load('bookable'); // Eager load the polymorphic relationship (SchoolService model)

        return response()->json([
            'booking' => [
                'id' => $booking->id,
                'school_service_name' => $booking->bookable->name,
                'booking_date' => $booking->booking_date,
                'booking_time' => $booking->booking_time,
                'status' => $booking->status,
                'notes' => $booking->notes,
                'school_service' => $booking->bookable,
            ]
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
