import React, { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { Swiper, SwiperSlide } from 'swiper/react';
import { Navigation, Pagination, Autoplay } from 'swiper/modules';
import {
    FaTimes,
    FaShoppingCart,
    FaStar,
    FaTruck,
    FaShieldAlt,
    FaUndo
} from 'react-icons/fa';
import { useForm } from '@inertiajs/react';
import { router } from '@inertiajs/react';

import 'swiper/css';
import 'swiper/css/navigation';
import 'swiper/css/pagination';

const AddToCartModal = ({ product, onClose }) => {
    const [currentImageIndex, setCurrentImageIndex] = useState(0);
    const [quantity, setQuantity] = useState(1);
    const [isAdded, setIsAdded] = useState(false);
    const [scrollY, setScrollY] = useState(0);

    const { data, setData, post, processing, reset } = useForm({
        item_id: product?.id || '',
        item_type: 'product',
        quantity: 1,
    });

    useEffect(() => {
        if (product) {
            setScrollY(window.scrollY);
            document.body.style.overflow = 'hidden';
        } else {
            document.body.style.overflow = 'unset';
            window.scrollTo(0, scrollY);
            reset(); // Reset the form when the modal closes
        }
    }, [product]);

    useEffect(() => {
        if (product) {
            setQuantity(1);
            setIsAdded(false);
            setData('item_id', product.id);
            setData('item_type', 'product');
            setData('quantity', 1);
        }
    }, [product]);

    if (!product) return null;

    const imageUrls = product.image_urls || [];
    const allImages = [...imageUrls];

    const handleAddToCartClick = () => {
        post(route('cart.add'), {
            onSuccess: () => {
                setIsAdded(true);
                setTimeout(() => {
                    setIsAdded(false);
                    onClose();
                }, 1500);
                console.log("fucked")
            },
            onError: (errors) => {
                console.error('Add to cart error:', errors);
            }
        });
    };

    const updateQuantity = (newQuantity) => {
        const validQuantity = Math.max(1, Math.min(10, newQuantity));
        setQuantity(validQuantity);
        setData('quantity', validQuantity);
    };

    return (
        <AnimatePresence>
            <motion.div
                className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                exit={{ opacity: 0 }}
                onClick={onClose}
            >
                <motion.div
                    className="bg-white rounded-xl shadow-xl max-w-md w-full max-h-[90vh] overflow-y-auto"
                    initial={{ scale: 0.9, opacity: 0 }}
                    animate={{ scale: 1, opacity: 1 }}
                    exit={{ scale: 0.9, opacity: 0 }}
                    onClick={(e) => e.stopPropagation()}
                >
                    {/* Header */}
                    <div className="flex items-center justify-between p-4 border-b">
                        <h2 className="text-xl font-semibold">Product Details</h2>
                        <button 
                            onClick={onClose}
                            className="text-gray-400 hover:text-gray-600"
                        >
                            <FaTimes className="h-5 w-5" />
                        </button>
                    </div>
                    
                    {/* Product Content */}
                    <div className="p-4">
                        {/* Product Image */}
                        <div className="relative mb-4">
                            <div className="relative h-56 bg-gray-100 rounded-lg overflow-hidden">
                                {allImages.length > 1 ? (
                                    <Swiper
                                        modules={[Navigation, Pagination, Autoplay]}
                                        navigation
                                        pagination={{ clickable: true }}
                                        autoplay={{ delay: 3000, disableOnInteraction: false }}
                                        loop={true}
                                        onSlideChange={(swiper) => setCurrentImageIndex(swiper.realIndex)}
                                        className="h-full"
                                    >
                                        {allImages.map((image, index) => (
                                            <SwiperSlide key={index}>
                                                <img 
                                                    src={image} 
                                                    alt={product.name} 
                                                    className="w-full h-full object-cover"
                                                    loading="lazy"
                                                />
                                            </SwiperSlide>
                                        ))}
                                    </Swiper>
                                ) : (
                                    <img 
                                        src={allImages[0]} 
                                        alt={product.name} 
                                        className="w-full h-full object-cover"
                                        loading="lazy"
                                    />
                                )}
                                
                                {allImages.length > 1 && (
                                    <div className="absolute bottom-2 right-2 bg-black/50 text-white text-xs px-2 py-1 rounded">
                                        {currentImageIndex + 1} / {allImages.length}
                                    </div>
                                )}
                            </div>
                        </div>
                        
                        {/* Product Info */}
                        <div className="mb-4">
                            <h3 className="text-lg font-semibold mb-2">{product.name}</h3>
                            
                            {/* Rating */}
                            <div className="flex items-center mb-3">
                                <div className="flex text-yellow-400 mr-2">
                                    {[...Array(5)].map((_, i) => (
                                        <FaStar key={i} className={`text-sm ${i < (product.rating || 0) ? "text-yellow-400" : "text-gray-300"}`} />
                                    ))}
                                </div>
                                <span className="text-sm text-gray-600">({product.reviews || 0} reviews)</span>
                            </div>
                            
                            {/* Price */}
                            <div className="flex items-baseline mb-3">
                                <span className="text-xl font-bold text-green-600">₵{product.price}</span>
                                {product.original_price && (
                                    <span className="ml-2 text-sm text-gray-500 line-through">₵{product.original_price}</span>
                                )}
                                {product.discount && (
                                    <span className="ml-2 bg-red-100 text-red-600 text-xs font-semibold px-2 py-1 rounded">
                                        -{product.discount}% OFF
                                    </span>
                                )}
                            </div>
                            
                            {/* Product Description */}
                            <p className="text-gray-600 text-sm mb-4">
                                {product.description}
                            </p>
                            
                            {/* Quantity Selector */}
                            <div className="flex items-center justify-between mb-4">
                                <span className="text-sm font-medium">Quantity:</span>
                                <div className="flex items-center border rounded-lg">
                                    <button 
                                        className="px-3 py-1 bg-gray-100 hover:bg-gray-200"
                                        onClick={() => updateQuantity(quantity - 1)}
                                    >
                                        -
                                    </button>
                                    <span className="px-4 py-1">{quantity}</span>
                                    <button 
                                        className="px-3 py-1 bg-gray-100 hover:bg-gray-200"
                                        onClick={() => updateQuantity(quantity + 1)}
                                    >
                                        +
                                    </button>
                                </div>
                            </div>
                            
                            {/* Total Price */}
                            <div className="flex justify-between mb-4">
                                <span className="text-sm font-medium">Total:</span>
                                <span className="text-lg font-semibold">₵{(product.price * quantity).toFixed(2)}</span>
                            </div>
                            
                            {/* Stock Status */}
                            <div className="flex items-center mb-4 text-sm text-gray-600">
                                <div className="w-2 h-2 bg-green-500 rounded-full mr-2"></div>
                                In Stock - Only {product.stock || 5} left
                            </div>
                        </div>
                        
                        {/* Action Buttons */}
                        <div className="flex space-x-3">
                            <button
                                onClick={onClose}
                                className="flex-1 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
                            >
                                Continue Shopping
                            </button>
                            <button
                                onClick={handleAddToCartClick}
                                disabled={processing}
                                className={`flex-1 px-4 py-2 rounded-lg text-white ${
                                    isAdded 
                                    ? 'bg-green-600' 
                                    : 'bg-green-600 hover:bg-green-700'
                                }`}
                            >
                                {isAdded ? 'Added!' : 'Add to Cart'}
                            </button>
                        </div>
                    </div>
                </motion.div>
            </motion.div>
        </AnimatePresence>
    );
};

export default AddToCartModal;