import React, { useState, useEffect } from 'react';
import { motion, AnimatePresence } from 'framer-motion';
import { useForm, Link, usePage } from '@inertiajs/react';
import axios from 'axios';
import Swal from 'sweetalert2';
import { showErrorToast, showSuccessToast } from './ToastNote';
import SuccessDisplay from './SuccessDisplay';
import { CalendarIcon, MapPinIcon, XMarkIcon, CreditCardIcon, ClockIcon, InformationCircleIcon, ArrowRightIcon, LockClosedIcon } from '@heroicons/react/24/outline';

// Helper function for currency formatting
const formatCurrency = (price) => {
    if (price === undefined || price === null) return 'N/A';
    return new Intl.NumberFormat('en-GH', {
        style: 'currency',
        currency: 'GHS',
        minimumFractionDigits: 2,
    }).format(price);
};

export default function BookingModal({ service, show, onClose }) {
    const { auth } = usePage().props;
    const isAuthenticated = !!auth.user;
    const [currentStep, setCurrentStep] = useState(1);
    // Fixed: Added isPaidService variable
    const isPaidService = service?.type === 'paid';
    const totalSteps = isAuthenticated ? (isPaidService ? 3 : 2) : 1;
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [showSuccess, setShowSuccess] = useState(false);
    const [successMessage, setSuccessMessage] = useState('');
    const [paystackAuthUrl, setPaystackAuthUrl] = useState(null);
    const [isIframeLoading, setIsIframeLoading] = useState(true);
    const [scrollY, setScrollY] = useState(0);

    useEffect(() => {
        if (service) {
            setScrollY(window.scrollY);
        } else {
            document.body.style.overflow = 'unset';
            window.scrollTo(0, scrollY);
            handleClose(); // Call handleClose to reset form and state when modal is dismissed
        }
    }, [service]);

    const { data, setData, post, processing, errors, reset } = useForm({
        service_id: service ? service.id : '',
        name: auth.user?.name || '',
        email: auth.user?.email || '',
        booking_date: '',
        booking_time: '',
        notes: '',
        terms: false,
        payment_method: '',
        quantity: 1
    });

    useEffect(() => {
        if (service) {
            setData('service_id', service.id);
        }

        const handleMessage = (event) => {
            try {
                const message = JSON.parse(event.data);
                if (message.status === 'success') {
                    handlePaymentSuccess(message.data);
                }
            } catch (error) {
                console.error('Error parsing postMessage data:', error);
            }
        };

        window.addEventListener('message', handleMessage);

        return () => {
            window.removeEventListener('message', handleMessage);
        };
    }, [service]);

    const submitBooking = async (e) => {
        e.preventDefault();
        setIsSubmitting(true);

        try {
            const response = await axios.post(route('bookings.store'), {
                bookable_type: 'App\\Models\\Service',
                bookable_id: service.id,
                price: service.price,
                notes: data.notes,
                name: auth.user?.name,
                email: auth.user?.email,
                booking_date: data.booking_date,
                booking_time: data.booking_time,
                quantity: 1
            });

            const { booking, message } = response.data;

            if (booking && isPaidService && data.payment_method) {
                try {
                    const paymentInitiationResponse = await axios.post(route('bookings.initiate-payment', booking.id), {
                        payment_method: data.payment_method,
                    });

                    if (paymentInitiationResponse.data.authorization_url) {
                        if (data.payment_method === 'paystack') {
                            setPaystackAuthUrl(paymentInitiationResponse.data.authorization_url);
                        } else if (data.payment_method === 'paypal') {
                            window.location.href = paymentInitiationResponse.data.authorization_url;
                        }
                    } else {
                        showErrorToast('Failed to get payment authorization URL.');
                    }
                } catch (err) {
                    showErrorToast('Failed to initiate payment process. Please try again.');
                }
            } else {
                setSuccessMessage(message || 'Your service booking has been confirmed!');
                setShowSuccess(true);
            }
        } catch (err) {
            showErrorToast('There was an error processing your booking. Please try again.');
        } finally {
            setIsSubmitting(false);
        }
    };

    const nextStep = () => {
        if (currentStep < totalSteps) {
            setCurrentStep(currentStep + 1);
        }
    };

    const prevStep = () => {
        if (currentStep > 1) {
            setCurrentStep(currentStep - 1);
        }
    };

    const handleClosePaymentIframe = () => {
        setPaystackAuthUrl(null);
        setIsIframeLoading(true);
        handleClose();
        Swal.fire({
            title: 'Payment Cancelled',
            text: 'Your payment was not completed. You can try again or select a different payment method.',
            icon: 'info',
            confirmButtonText: 'OK',
            confirmButtonColor: '#059669',
        });
    };

    const handlePaymentSuccess = (data) => {
        console.log("payment data", data)
        setSuccessMessage('Your booking has been successfully confirmed and payment processed!');
        setShowSuccess(true);
        setPaystackAuthUrl(null);
        setIsIframeLoading(true); // Reset iframe loading state
    };

    const handleClose = () => {
        setShowSuccess(false);
        setSuccessMessage('');
        setData('booking_date', '');
        setData('booking_time', '');
        setData('notes', '');
        setData('payment_method', '');
        setData('terms', false);
        reset(); // Resets useForm data
        setCurrentStep(1); // Reset to the first step
        onClose(); // Call the parent's onClose handler
    };

    if (!service) return null;

    // Progress indicator component
    const ProgressIndicator = () => (
        <div className="flex items-center justify-between mb-4 sm:mb-6">
            {[...Array(totalSteps)].map((_, i) => (
                <React.Fragment key={i}>
                    <div className="flex flex-col items-center">
                        <div className={`w-6 h-6 sm:w-8 sm:h-8 md:w-10 md:h-10 rounded-full flex items-center justify-center ${
                            i + 1 < currentStep ? 'bg-emerald-500 text-white' : 
                            i + 1 === currentStep ? 'bg-emerald-600 text-white shadow-lg' : 
                            'bg-gray-200 text-gray-500'
                        }`}>
                            {i + 1 < currentStep ? (
                                <svg className="w-3 h-3 sm:w-4 sm:h-4 md:w-6 md:h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 13l4 4L19 7" />
                                </svg>
                            ) : (
                                <span className="text-xs sm:text-sm font-medium">{i + 1}</span>
                            )}
                        </div>
                        <span className={`text-xs mt-1 hidden sm:block ${
                            i + 1 <= currentStep ? 'text-emerald-600 font-medium' : 'text-gray-500'
                        }`}>
                            {i === 0 ? 'Details' : i === 1 ? (isPaidService ? 'Payment' : 'Confirm') : 'Confirm'}
                        </span>
                    </div>
                    {i < totalSteps - 1 && (
                        <div className={`flex-1 h-1 mx-1 sm:mx-2 ${
                            i + 1 < currentStep ? 'bg-emerald-500' : 'bg-gray-200'
                        }`} />
                    )}
                </React.Fragment>
            ))}
        </div>
    );

    return (
        <AnimatePresence>
            {show && (
                <motion.div
                    className="fixed inset-0 bg-gray-900 bg-opacity-60 flex items-center justify-center z-50 p-2 sm:p-4"
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    exit={{ opacity: 0 }}
                    transition={{ duration: 0.3 }}
                >
                    <motion.div
                        className="bg-white rounded-xl sm:rounded-2xl shadow-2xl max-w-md w-full max-h-[90vh] overflow-hidden transform"
                        initial={{ scale: 0.9, opacity: 0 }}
                        animate={{ scale: 1, opacity: 1 }}
                        exit={{ scale: 0.9, opacity: 0 }}
                        transition={{ duration: 0.3, ease: "easeOut" }}
                    >
                        {
                            showSuccess ? (
                                <SuccessDisplay message={successMessage} onClose={handleClose} />
                            ) : (
                                <>
                                    <div className="relative bg-gradient-to-r from-emerald-500 to-teal-600 p-3 sm:p-4 md:p-6 text-white">
                                        <button
                                            onClick={paystackAuthUrl ? handleClosePaymentIframe : handleClose}
                                            className="absolute top-2 sm:top-3 md:top-4 right-2 sm:right-3 md:right-4 text-white/80 hover:text-white transition-colors p-1"
                                            aria-label="Close modal"
                                        >
                                            <XMarkIcon className="h-4 w-4 sm:h-5 sm:w-5 md:h-6 md:w-6" />
                                        </button>
                                        
                                        <div className="flex items-center space-x-2 sm:space-x-3">
                                            <div className="bg-white/20 p-2 sm:p-3 rounded-full backdrop-blur-sm">
                                                <CalendarIcon className="h-4 w-4 sm:h-6 sm:w-6 md:h-8 md:w-8" />
                                            </div>
                                            <div>
                                                <h2 className="text-base sm:text-lg md:text-2xl font-bold">{paystackAuthUrl ? 'Complete Your Payment' : 'Book Your Service'}</h2>
                                                <p className="text-emerald-100 text-xs sm:text-sm mt-1">Reserve your spot for this service</p>
                                            </div>
                                        </div>
                                    </div>

                                    <div className="px-3 sm:px-4 md:px-6 py-2 sm:py-3 md:py-4 bg-gradient-to-r from-emerald-50 to-teal-50 border-b border-emerald-100">
                                        <h3 className="text-sm sm:text-base md:text-lg font-semibold text-emerald-800 mb-1 sm:mb-2">{service.title}</h3>
                                        <div className="flex flex-wrap items-center text-xs sm:text-sm text-emerald-700 gap-2 sm:gap-4">
                                            <div className="flex items-center">
                                                <ClockIcon className="h-3 w-3 sm:h-4 sm:w-4 mr-1" />
                                                <span>{service.duration}</span>
                                            </div>
                                            {isPaidService && (
                                                <div className="flex items-center font-semibold">
                                                    <CreditCardIcon className="h-3 w-3 sm:h-4 sm:w-4 mr-1" />
                                                    <span>{formatCurrency(service.price)}</span>
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                    
                                    <div className="p-3 sm:p-4 md:p-6 overflow-y-auto max-h-[60vh]">
                                        {paystackAuthUrl ? (
                                            <div className="w-full h-[500px] sm:h-[600px] relative flex flex-col items-center justify-center">
                                                <p className="text-center text-gray-600 mb-4 font-medium text-sm sm:text-base px-2">
                                                    You are securely redirected to our payment partner **PayStack**. Please complete the transaction in the window below.
                                                </p>

                                                {isIframeLoading && (
                                                    <div className="absolute inset-0 bg-gray-50 flex flex-col items-center justify-center rounded-xl shadow-inner z-20">
                                                        <p className="text-gray-700 font-medium text-sm">Loading secure payment portal...</p>
                                                    </div>
                                                )}

                                                <iframe
                                                    src={paystackAuthUrl}
                                                    className={`w-full h-full border-2 border-gray-200 rounded-xl shadow-xl transition-opacity duration-500 ${isIframeLoading ? 'opacity-0' : 'opacity-100'}`}
                                                    title="Payment Gateway"
                                                    sandbox="allow-scripts allow-same-origin allow-forms allow-popups"
                                                    onLoad={() => setIsIframeLoading(false)}
                                                ></iframe>
                                            </div>
                                        ) : !isAuthenticated ? (
                                            <div className="text-center py-4 sm:py-6 md:py-10 px-3 sm:px-4 md:px-6">
                                                <div className="inline-flex items-center justify-center w-12 h-12 sm:w-16 sm:h-16 md:w-20 md:h-20 bg-red-100 rounded-full mb-3 sm:mb-4 md:mb-6 shadow-lg">
                                                    <LockClosedIcon className="h-6 w-6 sm:h-8 sm:w-8 md:h-10 md:w-10 text-red-600" />
                                                </div>
                                                <h3 className="text-lg sm:text-xl md:text-2xl font-bold text-gray-800 mb-2 sm:mb-3">Authentication Required</h3>
                                                <p className="text-gray-600 mb-4 sm:mb-6 md:mb-8 max-w-sm mx-auto text-sm sm:text-base">
                                                    Please log in or create an account to proceed with your event booking.
                                                </p>
                                                <div className="flex flex-col space-y-3">
                                                    <Link 
                                                        href={route('login')} 
                                                        className="flex items-center justify-center w-full px-3 sm:px-4 md:px-6 py-2 sm:py-3 bg-gradient-to-r from-emerald-500 to-teal-600 text-white rounded-lg font-semibold hover:from-emerald-600 hover:to-teal-700 transition-all shadow-md transform hover:scale-105 text-sm sm:text-base"
                                                    >
                                                        Log In <ArrowRightIcon className="h-4 w-4 sm:h-5 sm:w-5 ml-2" />
                                                    </Link>
                                                    <Link 
                                                        href={route('register')} 
                                                        className="flex items-center justify-center w-full px-3 sm:px-4 md:px-6 py-2 sm:py-3 border-2 border-gray-300 text-gray-700 rounded-lg font-semibold hover:bg-gray-50 transition-colors text-sm sm:text-base"
                                                    >
                                                        Create Account
                                                    </Link>
                                                </div>
                                            </div>
                                        ) : (
                                            <>
                                                {isAuthenticated && <ProgressIndicator />}
                                                
                                                <div className="space-y-3 sm:space-y-4 md:space-y-6">
                                                    {currentStep === 1 && (
                                                        <motion.div 
                                                            initial={{ opacity: 0, x: 20 }} 
                                                            animate={{ opacity: 1, x: 0 }} 
                                                            exit={{ opacity: 0, x: -20 }}
                                                            transition={{ duration: 0.3 }}
                                                        >
                                                            <div className="space-y-3 sm:space-y-4 md:space-y-5">   
                                                                <div>
                                                                    <label htmlFor="booking_date" className="flex items-center text-sm font-medium text-gray-700 mb-2">
                                                                        <CalendarIcon className="h-4 w-4 mr-1 text-emerald-500" />
                                                                        Preferred Date
                                                                    </label>
                                                                    <input 
                                                                        id="booking_date" 
                                                                        type="date"
                                                                        name="booking_date" 
                                                                        value={data.booking_date} 
                                                                        onChange={(e) => setData('booking_date', e.target.value)} 
                                                                        className="block w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 text-sm sm:text-base transition-colors"
                                                                        min={new Date().toISOString().split('T')[0]}
                                                                    />
                                                                    {errors.booking_date && <p className="mt-1 text-sm text-red-600 flex items-center"><InformationCircleIcon className="h-4 w-4 mr-1" />{errors.booking_date}</p>}
                                                                </div>
                                                                
                                                                <div>
                                                                    <label htmlFor="booking_time" className="flex items-center text-sm font-medium text-gray-700 mb-2">
                                                                        <ClockIcon className="h-4 w-4 mr-1 text-emerald-500" />
                                                                        Preferred Time
                                                                    </label>
                                                                    <input 
                                                                        id="booking_time" 
                                                                        type="time"
                                                                        name="booking_time" 
                                                                        value={data.booking_time} 
                                                                        onChange={(e) => setData('booking_time', e.target.value)} 
                                                                        className="block w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 text-sm sm:text-base transition-colors"
                                                                    />
                                                                    {errors.booking_time && <p className="mt-1 text-sm text-red-600 flex items-center"><InformationCircleIcon className="h-4 w-4 mr-1" />{errors.booking_time}</p>}
                                                                </div>

                                                                <div>
                                                                    <label htmlFor="notes" className="flex items-center text-sm font-medium text-gray-700 mb-2">
                                                                        <InformationCircleIcon className="h-4 w-4 mr-1 text-emerald-500" />
                                                                        Additional Notes (Optional)
                                                                    </label>
                                                                    <textarea
                                                                        id="notes"
                                                                        name="notes"
                                                                        rows="3"
                                                                        value={data.notes}
                                                                        className="block w-full px-3 sm:px-4 py-2 border border-gray-300 rounded-lg shadow-sm focus:outline-none focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500 text-sm sm:text-base transition-colors"
                                                                        onChange={(e) => setData('notes', e.target.value)}
                                                                        placeholder="Any special requirements or notes..."
                                                                    ></textarea>
                                                                    {errors.notes && <p className="mt-1 text-sm text-red-600 flex items-center"><InformationCircleIcon className="h-4 w-4 mr-1" />{errors.notes}</p>}
                                                                </div>
                                                            </div>
                                                        </motion.div>
                                                    )}

                                                    {currentStep === 2 && isPaidService && (
                                                        <motion.div 
                                                            initial={{ opacity: 0, x: 20 }} 
                                                            animate={{ opacity: 1, x: 0 }} 
                                                            exit={{ opacity: 0, x: -20 }}
                                                            transition={{ duration: 0.3 }}
                                                        >
                                                            <h3 className="text-base sm:text-lg md:text-xl font-semibold text-gray-800 mb-3 sm:mb-4">Payment Method</h3>
                                                            <div className="space-y-3 sm:space-y-4">
                                                                <div>
                                                                    <label className={`flex items-center p-3 sm:p-4 border-2 rounded-lg cursor-pointer hover:bg-gray-50 transition-all ${data.payment_method === 'paystack' ? 'border-emerald-500 bg-emerald-50' : 'border-gray-200'}`}>
                                                                        <input 
                                                                            type="radio" 
                                                                            name="payment_method" 
                                                                            value="paystack" 
                                                                            checked={data.payment_method === 'paystack'} 
                                                                            onChange={(e) => setData('payment_method', e.target.value)} 
                                                                            className="form-radio h-4 w-4 sm:h-5 sm:w-5 text-emerald-600" 
                                                                        />
                                                                        <div className="ml-3 sm:ml-4 flex items-center">
                                                                            <div className="w-8 h-8 sm:w-10 sm:h-10 bg-blue-100 rounded-full flex items-center justify-center mr-2 sm:mr-3">
                                                                                <CreditCardIcon className="h-4 w-4 sm:h-6 sm:w-6 text-blue-600" />
                                                                            </div>
                                                                            <div>
                                                                                <p className="text-sm sm:text-base font-medium text-gray-900">PayStack</p>
                                                                                <p className="text-xs sm:text-sm text-gray-500">Secure payment via PayStack</p>
                                                                            </div>
                                                                        </div>
                                                                    </label>
                                                                </div>
                                                                
                                                                <div>
                                                                    <label className={`flex items-center p-3 sm:p-4 border-2 rounded-lg cursor-pointer hover:bg-gray-50 transition-all ${data.payment_method === 'paypal' ? 'border-emerald-500 bg-emerald-50' : 'border-gray-200'}`}>
                                                                        <input 
                                                                            type="radio" 
                                                                            name="payment_method" 
                                                                            value="paypal" 
                                                                            checked={data.payment_method === 'paypal'} 
                                                                            onChange={(e) => setData('payment_method', e.target.value)} 
                                                                            className="form-radio h-4 w-4 sm:h-5 sm:w-5 text-emerald-600" 
                                                                        />
                                                                        <div className="ml-3 sm:ml-4 flex items-center">
                                                                            <div className="w-8 h-8 sm:w-10 sm:h-10 bg-yellow-100 rounded-full flex items-center justify-center mr-2 sm:mr-3">
                                                                                <CreditCardIcon className="h-4 w-4 sm:h-6 sm:w-6 text-yellow-600" />
                                                                            </div>
                                                                            <div>
                                                                                <p className="text-sm sm:text-base font-medium text-gray-900">PayPal</p>
                                                                                <p className="text-xs sm:text-sm text-gray-500">Pay with your PayPal account</p>
                                                                            </div>
                                                                        </div>
                                                                    </label>
                                                                </div>
                                                                {errors.payment_method && <p className="mt-1 text-sm text-red-600 flex items-center"><InformationCircleIcon className="h-4 w-4 mr-1" />{errors.payment_method}</p>}
                                                            </div>
                                                            
                                                            <div className="mt-3 sm:mt-4 md:mt-6 bg-gradient-to-r from-emerald-50 to-teal-50 rounded-lg sm:rounded-xl p-3 sm:p-4 md:p-5 border border-emerald-200">
                                                                <h4 className="text-sm sm:text-base font-semibold text-gray-800 mb-2 sm:mb-3 flex items-center">
                                                                    <CreditCardIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-emerald-600" />
                                                                    Payment Summary
                                                                </h4>
                                                                <div className="space-y-2 sm:space-y-3">
                                                                    <div className="flex justify-between text-sm">
                                                                        <span className="text-gray-600">Service:</span>
                                                                        <span className="font-medium text-gray-900">{service?.title}</span>
                                                                    </div>
                                                                    <div className="flex justify-between text-sm">
                                                                        <span className="text-gray-600">Price:</span>
                                                                        <span className="font-medium text-gray-900">{formatCurrency(service?.price)}</span>
                                                                    </div>
                                                                    <div className="pt-2 sm:pt-3 mt-2 sm:mt-3 border-t border-emerald-200 flex justify-between">
                                                                        <span className="font-semibold text-gray-800 text-sm">Total:</span>
                                                                        <span className="font-bold text-emerald-600 text-base sm:text-lg">{formatCurrency(service?.price)}</span>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                        </motion.div>
                                                    )}

                                                    {currentStep === totalSteps && (
                                                        <motion.div 
                                                            initial={{ opacity: 0, scale: 0.9 }} 
                                                            animate={{ opacity: 1, scale: 1 }} 
                                                            exit={{ opacity: 0, scale: 0.9 }}
                                                            transition={{ duration: 0.3 }}
                                                            className="text-center py-4 sm:py-6 md:py-10 px-3 sm:px-4 md:px-6"
                                                        >
                                                            <h3 className="text-lg sm:text-xl md:text-2xl font-bold text-gray-800 mb-2 sm:mb-3">Kindly confirm your booking</h3>
                                                            <p className="text-gray-600 mb-3 sm:mb-4 md:mb-6 text-sm sm:text-base"></p>
                                                            
                                                            <div className="bg-emerald-50 rounded-lg p-3 sm:p-4 border border-emerald-200 text-left">
                                                                <h4 className="font-semibold text-gray-800 mb-2 text-sm sm:text-base">Booking Details:</h4>
                                                                <div className="space-y-1 text-xs sm:text-sm">
                                                                    <div className="flex justify-between">
                                                                        <span className="text-gray-600">Service:</span>
                                                                        <span className="font-medium text-gray-900">{service?.title}</span>
                                                                    </div>
                                                                    <div className="flex justify-between">
                                                                        <span className="text-gray-600">Date:</span>
                                                                        <span className="font-medium text-gray-900">{data.booking_date}</span>
                                                                    </div>
                                                                    <div className="flex justify-between">
                                                                        <span className="text-gray-600">Time:</span>
                                                                        <span className="font-medium text-gray-900">{data.booking_time}</span>
                                                                    </div>
                                                                    {isPaidService && (
                                                                        <div className="flex justify-between">
                                                                            <span className="text-gray-600">Total Paid:</span>
                                                                            <span className="font-medium text-gray-900">{formatCurrency(service?.price)}</span>
                                                                        </div>
                                                                    )}
                                                                </div>
                                                            </div>
                                                        </motion.div>
                                                    )}

                                                    <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center pt-3 sm:pt-4 md:pt-6 border-t border-gray-200 gap-3">
                                                        <button 
                                                            type="button" 
                                                            onClick={onClose} 
                                                            className="px-4 py-2 text-gray-600 hover:text-gray-800 transition-colors text-sm sm:text-base order-2 sm:order-1"
                                                        >
                                                            Cancel
                                                        </button>
                                                        <div className="flex justify-between sm:justify-end space-x-3 order-1 sm:order-2">
                                                            {currentStep > 1 && (
                                                                <button 
                                                                    type="button" 
                                                                    disabled={isSubmitting}
                                                                    onClick={prevStep}  // Fixed: Changed from handlePrevStep to prevStep
                                                                    className="px-3 sm:px-4 md:px-5 py-2 border border-gray-300 rounded-lg shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500 transition-all"
                                                                >
                                                                    Previous
                                                                </button>
                                                            )}
                                                            {currentStep < totalSteps ? (
                                                                <button 
                                                                    type="button" 
                                                                    disabled={isSubmitting}
                                                                    onClick={nextStep}  // Fixed: Changed from handleNextStep to nextStep
                                                                    className="px-3 sm:px-4 md:px-6 py-2 border border-transparent rounded-lg shadow-md text-sm font-medium text-white bg-gradient-to-r from-emerald-500 to-teal-600 hover:from-emerald-600 hover:to-teal-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500 disabled:opacity-50 disabled:cursor-not-allowed transition-all flex items-center justify-center min-w-[80px] sm:min-w-[100px] md:min-w-[120px]"
                                                                >
                                                                    Next <ArrowRightIcon className="h-4 w-4 ml-2" />
                                                                </button>
                                                            ) : (
                                                                <button 
                                                                    type="submit" 
                                                                    onClick={submitBooking}
                                                                    disabled={isSubmitting} 
                                                                    className="px-3 sm:px-4 md:px-6 py-2 border border-transparent rounded-lg shadow-md text-sm font-medium text-white bg-gradient-to-r from-emerald-500 to-teal-600 hover:from-emerald-600 hover:to-teal-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500 disabled:opacity-50 disabled:cursor-not-allowed transition-all flex items-center justify-center min-w-[80px] sm:min-w-[100px] md:min-w-[120px]"
                                                                >
                                                                    {isSubmitting ? (
                                                                        <>
                                                                            <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                                                <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                                                <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                                            </svg>
                                                                            Processing...
                                                                        </>
                                                                    ) : (
                                                                        'Confirm Booking'
                                                                    )}
                                                                </button>
                                                            )}
                                                        </div>
                                                    </div>
                                                </div>
                                            </>
                                        )}
                                    </div>
                                </>
                            )
                        }
                    </motion.div>
                </motion.div>
            )}
        </AnimatePresence>
    );
}