import React from 'react';
import { motion } from 'framer-motion';
import { FaFacebook, FaTwitter, FaLinkedin, FaInstagram, FaArrowUp } from 'react-icons/fa';
import { FiMail, FiPhone, FiMapPin } from 'react-icons/fi';

// Animation variants
const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.1,
            delayChildren: 0.2
        }
    }
};

const itemVariants = {
    hidden: { y: 20, opacity: 0 },
    visible: {
        y: 0,
        opacity: 1,
        transition: { duration: 0.5, ease: "easeOut" }
    }
};

const Footer = ({ setting }) => {
    // Quick links data
    const quickLinks = [
        { name: 'About Us', href: '#about' },
        { name: 'Our Services', href: '#services' },
        { name: 'Portfolio', href: '#portfolio' },
        { name: 'Our Team', href: '#team' },
        { name: 'Contact', href: '#contact' }
    ];

    // Contact information
    const contactInfo = [
        { icon: <FiMapPin className="text-lg" />, text: setting ? setting?.address : '123 Main St, Anytown, USA' },
        { icon: <FiPhone className="text-lg" />, text: setting ? setting?.phone : '+1 (555) 123-4567' },
        { icon: <FiMail className="text-lg" />, text: setting ? setting?.email : 'hello@promisebalm.com' }
    ];

    // Social media links
    const socialLinks = [
        { name: 'Facebook', icon: <FaFacebook />, href: setting ? setting?.facebook_link : '#' },
        { name: 'Twitter', icon: <FaTwitter />, href: setting ? setting?.twitter_link : '#' },
        { name: 'LinkedIn', icon: <FaLinkedin />, href: setting ? setting?.linkedin_link : '#' },
        { name: 'Instagram', icon: <FaInstagram />, href: setting ? setting?.instagram_link : '#' }
    ];

    // Scroll to top function
    const scrollToTop = () => {
        window.scrollTo({
            top: 0,
            behavior: 'smooth'
        });
    };

    return (
        <footer className="bg-gradient-to-b from-gray-900 to-gray-800 text-white pt-16 pb-8 relative overflow-hidden">
            {/* Decorative elements */}
            <div className="absolute top-0 left-0 w-full h-1 bg-gradient-to-r from-green-500 to-emerald-500"></div>
            <div className="absolute top-20 right-10 w-24 h-24 rounded-full bg-gradient-to-r from-green-500 to-emerald-500 opacity-10 blur-xl"></div>
            <div className="absolute bottom-20 left-10 w-32 h-32 rounded-full bg-gradient-to-r from-green-500 to-emerald-500 opacity-10 blur-xl"></div>
            
            <div className="container mx-auto px-6 relative z-10">
                <motion.div 
                    className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-10 mb-12"
                    variants={containerVariants}
                    initial="hidden"
                    whileInView="visible"
                    viewport={{ once: true, amount: 0.2 }}
                >
                    {/* Company Info */}
                    <motion.div variants={itemVariants}>
                        <div className="flex items-center mb-6">
                            <div className="w-12 h-12 rounded-xl bg-gradient-to-br from-green-500 to-emerald-500 flex items-center justify-center shadow-lg mr-4">
                                <svg className="h-6 w-6 text-white" fill="currentColor" viewBox="0 0 20 20">
                                    <path fillRule="evenodd" d="M5.05 4.05a7 7 0 119.9 9.9L10 18.9l-4.95-4.95a7 7 0 010-9.9zM10 11a2 2 0 100-4 2 2 0 000 4z" clipRule="evenodd" />
                                </svg>
                            </div>
                            <div>
                                <h3 className="text-2xl font-bold">Promise</h3>
                                <h3 className="text-2xl font-bold text-green-400">Balm</h3>
                            </div>
                        </div>
                        <p className="text-gray-300 mb-6">
                            We're dedicated to providing quality healthcare services and wellness solutions. 
                            Our team of experts is committed to helping you on your journey to better health.
                        </p>
                    </motion.div>
                    
                    {/* Quick Links */}
                    <motion.div variants={itemVariants}>
                        <h3 className="text-xl font-bold mb-6 pb-2 relative inline-block">
                            Quick Links
                            <span className="absolute bottom-0 left-0 w-full h-0.5 bg-gradient-to-r from-green-500 to-emerald-500 rounded-full"></span>
                        </h3>
                        <ul className="space-y-3">
                            {quickLinks.map((link, index) => (
                                <li key={index}>
                                    <a 
                                        href={link.href} 
                                        className="text-gray-300 hover:text-green-300 transition-colors duration-300 flex items-center"
                                    >
                                        <span className="w-1.5 h-1.5 rounded-full bg-green-500 mr-2"></span>
                                        {link.name}
                                    </a>
                                </li>
                            ))}
                        </ul>
                    </motion.div>
                    
                    {/* Contact Information */}
                    <motion.div variants={itemVariants}>
                        <h3 className="text-xl font-bold mb-6 pb-2 relative inline-block">
                            Contact Us
                            <span className="absolute bottom-0 left-0 w-full h-0.5 bg-gradient-to-r from-green-500 to-emerald-500 rounded-full"></span>
                        </h3>
                        <ul className="space-y-4">
                            {contactInfo.map((info, index) => (
                                <li key={index} className="flex items-start">
                                    <span className="text-green-500 mr-3 mt-1">{info.icon}</span>
                                    <span className="text-gray-300">{info.text}</span>
                                </li>
                            ))}
                        </ul>
                    </motion.div>
                    
                    {/* Social Media */}
                    <motion.div variants={itemVariants}>
                        <h3 className="text-xl font-bold mb-6 pb-2 relative inline-block">
                            Connect With Us
                            <span className="absolute bottom-0 left-0 w-full h-0.5 bg-gradient-to-r from-green-500 to-emerald-500 rounded-full"></span>
                        </h3>
                        <p className="text-gray-300 mb-6">
                            Follow us on social media for health tips, updates, and wellness insights.
                        </p>
                        <div className="flex space-x-4 mb-8">
                            {socialLinks.map((social, index) => (
                                <motion.a
                                    key={index}
                                    href={social.href}
                                    className="w-12 h-12 rounded-full bg-gray-700 flex items-center justify-center text-gray-300 hover:bg-gradient-to-r hover:from-green-500 hover:to-emerald-500 hover:text-white transition-all duration-300"
                                    aria-label={social.name}
                                    whileHover={{ 
                                        scale: 1.1,
                                        rotate: 5,
                                        backgroundColor: ["#374151", "#10b981", "#059669"]
                                    }}
                                    whileTap={{ scale: 0.9 }}
                                >
                                    {social.icon}
                                </motion.a>
                            ))}
                        </div>
                        
                        {/* Business Hours */}
                        <div className="bg-gray-700/50 p-4 rounded-lg">
                            <h4 className="font-semibold mb-2">Business Hours</h4>
                            <p className="text-gray-300 text-sm">Monday - Friday: 9:00 AM - 6:00 PM</p>
                            <p className="text-gray-300 text-sm">Saturday: 10:00 AM - 4:00 PM</p>
                            <p className="text-gray-300 text-sm">Sunday: Closed</p>
                        </div>
                    </motion.div>
                </motion.div>
                
                {/* Copyright and Back to Top */}
                <div className="border-t border-gray-700 pt-8">
                    <div className="flex flex-col md:flex-row justify-between items-center">
                        <motion.div 
                            className="text-gray-400 text-sm mb-4 md:mb-0"
                            initial={{ opacity: 0 }}
                            whileInView={{ opacity: 1 }}
                            transition={{ duration: 0.5 }}
                        >
                            &copy; {new Date().getFullYear()} Promise Balm. All rights reserved.
                        </motion.div>
                        
                        <motion.div
                            initial={{ opacity: 0 }}
                            whileInView={{ opacity: 1 }}
                            transition={{ duration: 0.5, delay: 0.2 }}
                        >
                            <motion.button
                                onClick={scrollToTop}
                                className="flex items-center text-green-500 hover:text-green-400 transition-colors duration-300"
                                whileHover={{ y: -3 }}
                                whileTap={{ scale: 0.95 }}
                            >
                                <span>Back to Top</span>
                                <FaArrowUp className="ml-2" />
                            </motion.button>
                        </motion.div>
                    </div>
                </div>
            </div>
        </footer>
    );
};

export default Footer;