import { Fragment, useEffect } from 'react';
import { Dialog, Transition } from '@headlessui/react';

// The main Modal component
export default function Modal({ children, show = false, maxWidth = '2xl', closeable = true, onClose = () => {} }) {
    const close = () => {
        if (closeable) {
            onClose();
        }
    };

    useEffect(() => {
        if (show) {
            document.body.style.overflow = 'hidden';
        } else {
            document.body.style.overflow = 'unset';
        }
    }, [show]);

    // Defines the max-width class based on the prop
    const maxWidthClass = {
        sm: 'sm:max-w-sm',
        md: 'sm:max-w-md',
        lg: 'sm:max-w-lg',
        xl: 'sm:max-w-xl',
        '2xl': 'sm:max-w-2xl',
    }[maxWidth];

    return (
        <Transition show={show} as={Fragment} leave="duration-200">
            <Dialog
                as="div"
                id="modal"
                // Main container with fixed positioning, centered content, and high Z-index
                className="fixed inset-0 flex overflow-y-auto px-4 py-6 sm:px-0 items-center justify-center z-[100] transform transition-all" 
                onClose={close}
            >
                {/* 1. Backdrop (Overlay) - Green Theme */}
                <Transition.Child
                    as={Fragment}
                    enter="ease-out duration-300"
                    enterFrom="opacity-0"
                    enterTo="opacity-100"
                    leave="ease-in duration-200"
                    leaveFrom="opacity-100"
                    leaveTo="opacity-0"
                >
                    {/* Dark green, transparent backdrop with blur */}
                    <div className="absolute inset-0 bg-emerald-900/75 backdrop-blur-sm" />
                </Transition.Child>

                {/* 2. Modal Content Container - Enhanced Transitions */}
                <Transition.Child
                    as={Fragment}
                    enter="ease-out duration-300"
                    enterFrom="opacity-0 translate-y-8 sm:translate-y-0 sm:scale-90"
                    enterTo="opacity-100 translate-y-0 sm:scale-100"
                    leave="ease-in duration-200"
                    leaveFrom="opacity-100 translate-y-0 sm:scale-100"
                    leaveTo="opacity-0 translate-y-8 sm:translate-y-0 sm:scale-90"
                >
                    {/* Dialog Panel - Attractive Styling with Green Border and Shadow */}
                    <Dialog.Panel
                        className={`bg-white rounded-xl overflow-hidden shadow-2xl border-t-4 border-emerald-500 transform transition-all sm:w-full sm:mx-auto ${maxWidthClass} my-6`}
                    >
                        {children}
                    </Dialog.Panel>
                </Transition.Child>
            </Dialog>
        </Transition>
    );
}

// Note: If you want to nest components (like a Header and Body), you can optionally add them here
// as sub-components for a clearer structure.