import React from 'react';
import { motion } from 'framer-motion';
import 'animate.css';

const SuccessDisplay = ({ message, onClose }) => {
    // Framer motion animation variants
    const containerVariants = {
        hidden: { 
            opacity: 0, 
            scale: 0.9,
            y: 20
        },
        visible: { 
            opacity: 1, 
            scale: 1,
            y: 0,
            transition: {
                duration: 0.5,
                ease: "easeOut",
                staggerChildren: 0.2
            }
        },
        exit: {
            opacity: 0,
            scale: 0.9,
            y: -20,
            transition: {
                duration: 0.3,
                ease: "easeIn"
            }
        }
    };

    const itemVariants = {
        hidden: { opacity: 0, y: 20 },
        visible: { 
            opacity: 1, 
            y: 0,
            transition: {
                duration: 0.4,
                ease: "easeOut"
            }
        }
    };

    const checkmarkVariants = {
        hidden: { 
            pathLength: 0,
            opacity: 0
        },
        visible: {
            pathLength: 1,
            opacity: 1,
            transition: {
                duration: 0.8,
                ease: "easeInOut",
                delay: 0.5
            }
        }
    };

    return (
        <motion.div
            className="bg-white p-10 rounded-lg shadow-xl max-w-md mx-auto border border-gray-200 relative overflow-hidden"
            variants={containerVariants}
            initial="hidden"
            animate="visible"
            exit="exit"
        >
            {/* Subtle background pattern for elegance */}
            <div className="absolute inset-0 opacity-5">
                <div className="absolute inset-0" style={{ 
                    backgroundImage: `url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23000000' fill-opacity='1'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E")` 
                }}></div>
            </div>
            
            <motion.div 
                className="text-center mb-8 relative z-10"
                variants={itemVariants}
            >
                {/* Animated success icon with green theme */}
                <div className="w-20 h-20 bg-gradient-to-br from-green-50 to-green-100 rounded-full flex items-center justify-center mx-auto mb-6 border border-green-100 shadow-sm animate__animated animate__pulse animate__infinite">
                    <div className="w-12 h-12 bg-gradient-to-br from-green-500 to-green-600 rounded-full flex items-center justify-center shadow-md">
                        <svg className="w-7 h-7 text-white" fill="none" strokeLinecap="round" strokeLinejoin="round" strokeWidth="2.5" viewBox="0 0 24 24" stroke="currentColor">
                            <motion.path 
                                d="M5 13l4 4L19 7" 
                                variants={checkmarkVariants}
                                initial="hidden"
                                animate="visible"
                            />
                        </svg>
                    </div>
                </div>
                
                <motion.h2 
                    className="text-2xl font-serif text-gray-800 mb-3"
                    variants={itemVariants}
                >
                    Booking Confirmed
                </motion.h2>
                <motion.p 
                    className="text-gray-600 leading-relaxed"
                    variants={itemVariants}
                >
                    {message || 'Your booking has been successfully processed.'}
                </motion.p>
            </motion.div>

            {/* Elegant divider with animation */}
            <motion.div 
                className="relative z-10 mb-6"
                variants={itemVariants}
            >
                <div className="h-px bg-gradient-to-r from-transparent via-green-200 to-transparent"></div>
            </motion.div>

            {/* Animated button with green theme */}
            <motion.div 
                className="text-center relative z-10"
                variants={itemVariants}
                whileHover={{ scale: 1.02 }}
                whileTap={{ scale: 0.98 }}
            >
                <button 
                    onClick={onClose} 
                    className="w-full bg-gradient-to-r from-green-600 to-green-700 hover:from-green-700 hover:to-green-800 text-white font-medium py-3 px-6 rounded-md transition-all duration-200 shadow-md hover:shadow-lg transform hover:-translate-y-0.5 animate__animated animate__fadeInUp"
                >
                    Continue
                </button>
                <motion.p 
                    className="text-xs text-gray-500 mt-4"
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ delay: 1.2, duration: 0.5 }}
                >
                    Confirmation details have been sent to your email
                </motion.p>
            </motion.div>

            {/* Floating particles animation */}
            <div className="absolute top-0 left-0 w-full h-full pointer-events-none overflow-hidden">
                {[...Array(6)].map((_, i) => (
                    <motion.div
                        key={i}
                        className="absolute w-2 h-2 bg-green-400 rounded-full opacity-30"
                        style={{
                            left: `${Math.random() * 100}%`,
                            top: `${Math.random() * 100}%`,
                        }}
                        animate={{
                            y: [0, -100, 0],
                            x: [0, Math.random() * 40 - 20, 0],
                            opacity: [0, 0.5, 0],
                        }}
                        transition={{
                            duration: 3 + Math.random() * 2,
                            repeat: Infinity,
                            delay: Math.random() * 2,
                            ease: "easeInOut"
                        }}
                    />
                ))}
            </div>
        </motion.div>
    );
};

export default SuccessDisplay;