import React, { useState } from 'react';
import { Link, usePage, router } from '@inertiajs/react';
import ApplicationLogo from '@/Components/ApplicationLogo';
import Navigation from './Navigation';
import FlashMessage from '@/Components/FlashMessage';

const formatDateTime = (dateString) => {
    if (!dateString) return 'N/A';
    
    const date = new Date(dateString);
    const options = { 
        year: 'numeric', 
        month: 'short', 
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    };
    return date.toLocaleDateString('en-US', options);
};

export default function AdminLayout({ children }) {
    const { auth } = usePage().props;
    const [sidebarOpen, setSidebarOpen] = useState(false);
    const [notificationsOpen, setNotificationsOpen] = useState(false);
    const [profileOpen, setProfileOpen] = useState(false);

    return (
        <div className="flex min-h-screen bg-gray-50">
            {/* Sidebar Backdrop for Mobile */}
            {sidebarOpen && (
                <div
                    className="fixed inset-0 z-20 bg-gray-900 bg-opacity-50 lg:hidden"
                    onClick={() => setSidebarOpen(false)}
                />
            )}

            {/* Sidebar */}
            <div className={`fixed inset-y-0 left-0 w-64 bg-gradient-to-b from-green-700 to-green-900 text-white transform ${sidebarOpen ? 'translate-x-0' : '-translate-x-full'} lg:translate-x-0 transition-transform duration-300 ease-in-out z-30 shadow-2xl`}>
                
                <Navigation />
            </div>

            {/* Content area */}
            <div className="flex-1 flex flex-col lg:ml-64">
                {/* Header for mobile (to open sidebar) */}
                <header className="fixed top-0 left-0 right-0 lg:hidden z-10 flex items-center justify-between px-4 py-3 bg-white shadow-md border-b border-green-100">
                    <button 
                        onClick={() => setSidebarOpen(!sidebarOpen)} 
                        className="text-green-600 p-2 rounded-md hover:bg-green-50 focus:outline-none focus:ring-2 focus:ring-inset focus:ring-green-500 transition-all duration-200"
                    >
                        <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth="2" d="M4 6h16M4 12h16M4 18h16" />
                        </svg>
                    </button>
                    <ApplicationLogo className="block h-8 w-auto fill-current text-green-600" />
                </header>

                {/* Desktop Header */}
                <header className="fixed top-0 left-64 right-0 hidden lg:flex items-center justify-between px-6 py-4 bg-white shadow-md border-b border-green-100 z-10">
                    <div className="flex items-center">
                        <h1 className="text-xl font-semibold text-gray-800">Admin Dashboard</h1>
                        <div className="ml-4 h-6 w-px bg-gray-300"></div>
                        <div className="ml-4 flex items-center text-sm text-gray-500">
                            <svg className="h-4 w-4 mr-1 text-green-500" fill="currentColor" viewBox="0 0 20 20">
                                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm1-12a1 1 0 10-2 0v4a1 1 0 00.293.707l2.828 2.829a1 1 0 101.415-1.415L11 9.586V6z" clipRule="evenodd" />
                            </svg>
                            Last login: {auth.last_login_at ? formatDateTime(auth.last_login_at) : 'N/A'}
                        </div>
                    </div>
                    <div className="flex items-center space-x-4">
                        <div className="relative">
                            <button
                                onClick={() => setNotificationsOpen(!notificationsOpen)}
                                className="relative p-1 rounded-full text-gray-400 hover:text-green-500 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 transition-colors"
                            >
                                <svg className="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9" />
                                </svg>
                                {auth.unreadNotifications.length > 0 && (
                                    <span className="absolute top-0 right-0 block h-2 w-2 rounded-full bg-green-400 ring-2 ring-white"></span>
                                )}
                            </button>
                            {notificationsOpen && (
                                <div className="absolute right-0 mt-2 w-80 bg-white rounded-lg shadow-xl overflow-hidden z-10">
                                    <div className="py-2 px-4 text-sm font-medium text-gray-700 border-b">
                                        Notifications ({auth.unreadNotifications.length})
                                    </div>
                                    <div className="divide-y max-h-60 overflow-y-auto">
                                        {auth.unreadNotifications.length > 0 ? (
                                            auth.unreadNotifications.map(notification => (
                                                <div key={notification.id} className="p-4 text-sm text-gray-700">
                                                    <p>{notification.data.message}</p>
                                                    <div className="flex justify-between items-center mt-2">
                                                        <span className="text-xs text-gray-500">{formatDateTime(notification.created_at)}</span>
                                                        <button
                                                            onClick={() => router.post(route('admin.notifications.markAsRead', notification.id))}
                                                            className="text-xs text-green-600 hover:text-green-800"
                                                        >
                                                            Mark as read
                                                        </button>
                                                    </div>
                                                </div>
                                            ))
                                        ) : (
                                            <div className="p-4 text-sm text-gray-500">
                                                No new notifications.
                                            </div>
                                        )}
                                    </div>
                                </div>
                            )}
                        </div>
                        <div className="relative">
                            <button
                                onClick={() => setProfileOpen(!profileOpen)}
                                className="flex items-center text-sm rounded-full focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                            >
                                <div className="h-8 w-8 rounded-full bg-gradient-to-r from-green-400 to-green-600 flex items-center justify-center shadow-md">
                                    <span className="text-white font-medium">{auth.user.name.charAt(0)}</span>
                                </div>
                            </button>
                            {profileOpen && (
                                <div className="absolute right-0 mt-2 w-48 bg-white rounded-md shadow-xl z-10">
                                    <div className="py-1">
                                        <Link
                                            href={route('profile.edit')}
                                            className="block px-4 py-2 text-sm text-gray-700 hover:bg-gray-100"
                                        >
                                            Edit Profile
                                        </Link>
                                        <button
                                            onClick={() => router.post(route('logout'))}
                                            className="block w-full text-left px-4 py-2 text-sm text-gray-700 hover:bg-gray-100"
                                        >
                                            Logout
                                        </button>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                </header>

                {/* Main Content - Fixed height with scroll */}
                <main className="fixed inset-0 pt-16 lg:pt-20 lg:left-64 lg:right-0 overflow-y-auto bg-gray-50">
                    {children}
                </main>
            </div>
        </div>
    );
}