import React, { useEffect, useState } from 'react';
import { Head } from '@inertiajs/react';
import Navbar from '@/Components/Navbar';
import Footer from '@/Components/Footer';
import { motion } from 'framer-motion';
import {
    FaLightbulb, FaUsers, FaChartLine,
    FaAward, FaRocket, FaHandshake, FaImages
} from 'react-icons/fa';
import Preloader from '@/Components/Preloader';
import GalleryItems from '@/Components/GalleryItems';

const PRIMARY_COLOR = '#CC0000';
const ACCENT_COLOR = '#FFD700';

const About = ({ setting, galleryItems }) => {
    const [loading, setLoading] = useState(true);
    const [selectedImage, setSelectedImage] = useState(null);
    
    useEffect(() => {
        const timer = setTimeout(() => {
            setLoading(false);
        }, 1000);

        return () => clearTimeout(timer);
    }, []);

    if (loading) {
        return <Preloader />;
    }
    
    const features = [
        {
            icon: <FaLightbulb className="text-3xl" />,
            title: "Innovation",
            description: setting ? setting?.innovation : "We embrace creativity to deliver cutting-edge solutions tailored to your needs."
        },
        {
            icon: <FaUsers className="text-3xl" />,
            title: "Teamwork",
            description: setting ? setting?.teamwork : "We foster collaboration to bring together diverse skills and perspectives."
        },
        {
            icon: <FaChartLine className="text-3xl" />,
            title: "Growth",
            description: setting ? setting?.growth : "Our mission is to create sustainable growth for clients and partners."
        },
        {
            icon: <FaAward className="text-3xl" />,
            title: "Excellence",
            description: setting ? setting?.excellence : "We are committed to quality, precision, and continuous improvement."
        },
        {
            icon: <FaRocket className="text-3xl" />,
            title: "Speed",
            description: setting ? setting?.speed : "We execute swiftly, ensuring timely delivery without compromising standards."
        },
        {
            icon: <FaHandshake className="text-3xl" />,
            title: "Integrity",
            description: setting ? setting?.integrity : "We build lasting trust through transparency and ethical practices."
        }
    ];

    // Gallery images - in a real app, these would come from props or an API
    const galleryImages = galleryItems.map(item => ({
        id: item.id,
        title: item.title,
        category: item.description,
        url: item.url
    }));

    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: { staggerChildren: 0.2, delayChildren: 0.3 }
        }
    };

    const itemVariants = {
        hidden: { y: 20, opacity: 0 },
        visible: { y: 0, opacity: 1, transition: { duration: 0.5 } }
    };

    return (
        <div className="min-h-screen bg-white font-sans mt-24 antialiased">
            <Head title="About Us" />
            <Navbar setting={setting} />

            {/* Hero Section - Matching the provided structure exactly */}
            <div className="relative bg-gradient-to-r from-red-700 to-red-800 py-16 text-white">
                <div className="container mx-auto px-6">
                    <div className="max-w-3xl">
                        <h1 className="text-4xl md:text-5xl font-bold mb-4">About {setting ? setting?.company_name : "Our Company"}</h1>
                        <p className="text-xl text-red-100 max-w-2xl">
                            Learn more about our values, mission, and commitment to excellence. We are dedicated to delivering innovative solutions that drive success.
                        </p>
                    </div>
                </div>
                
                {/* Subtle decorative elements */}
                <div className="absolute top-0 right-0 w-96 h-96 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
            </div>

            <section className="relative py-20 bg-gradient-to-b from-white to-gray-50" id="about">
                <div className="container mx-auto px-6 relative z-10">
                    {/* Company Description */}
                    <motion.div
                        className="max-w-4xl mx-auto text-center mb-20"
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.7, delay: 0.2 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl font-bold text-gray-800 mb-6">Our Story</h2>
                        <p className="text-lg leading-relaxed text-gray-600">
                            {setting ? setting?.about_us_description : "EK Company is a forward-thinking firm dedicated to delivering innovative solutions that drive success. Our team of experts collaborates closely with clients to understand their unique challenges and craft strategies that foster growth and excellence."}
                        </p>
                    </motion.div>

                    {/* Core Values Section */}
                    <motion.div
                        className="mb-16"
                        initial={{ opacity: 0 }}
                        whileInView={{ opacity: 1 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl font-bold text-center text-gray-800 mb-12">Our Core Values</h2>
                        
                        <motion.div
                            className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                            variants={containerVariants}
                            initial="hidden"
                            whileInView="visible"
                            viewport={{ once: true }}
                        >
                            {features.map((feature, index) => (
                                <motion.div
                                    key={index}
                                    className="bg-white rounded-xl p-8 shadow-md hover:shadow-lg border border-gray-100 transition-all duration-300 hover:-translate-y-1"
                                    variants={itemVariants}
                                    whileHover={{ scale: 1.02 }}
                                >
                                    <div
                                        className="w-16 h-16 rounded-full flex items-center justify-center bg-gradient-to-r from-red-600 to-yellow-500 text-white shadow-lg mb-6 mx-auto"
                                    >
                                        {feature.icon}
                                    </div>
                                    <h3 className="text-xl font-semibold text-gray-900 mb-3 text-center">{feature.title}</h3>
                                    <p className="text-gray-600 text-center">{feature.description}</p>
                                </motion.div>
                            ))}
                        </motion.div>
                    </motion.div>

                    {/* Statistics Section */}
                    <motion.div
                        className="bg-gradient-to-r from-red-700 to-red-800 rounded-2xl p-10 md:p-16 text-white shadow-xl overflow-hidden mb-20"
                        initial={{ opacity: 0, scale: 0.95 }}
                        whileInView={{ opacity: 1, scale: 1 }}
                        transition={{ duration: 0.7 }}
                        viewport={{ once: true }}
                    >
                        {/* Decorative elements */}
                        <div className="absolute top-0 right-0 w-64 h-64 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
                        <div className="absolute bottom-0 left-0 w-64 h-64 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
                        
                        <div className="relative z-10">
                            <h2 className="text-3xl font-bold text-center mb-12">Our Impact in Numbers</h2>
                            
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-10 text-center">
                                <div>
                                    <motion.div
                                        className="text-5xl font-bold mb-2"
                                        initial={{ opacity: 0 }}
                                        whileInView={{ opacity: 1 }}
                                        transition={{ duration: 0.5, delay: 0.2 }}
                                        viewport={{ once: true }}
                                    >
                                        {setting ? setting?.years_of_experience : "10"}+
                                    </motion.div>
                                    <p className="text-yellow-300 text-sm uppercase tracking-wide font-medium">Years of Experience</p>
                                </div>

                                <div>
                                    <motion.div
                                        className="text-5xl font-bold mb-2"
                                        initial={{ opacity: 0 }}
                                        whileInView={{ opacity: 1 }}
                                        transition={{ duration: 0.5, delay: 0.4 }}
                                        viewport={{ once: true }}
                                    >
                                        {setting ? setting?.happy_clients : "500"}+
                                    </motion.div>
                                    <p className="text-yellow-300 text-sm uppercase tracking-wide font-medium">Happy Clients</p>
                                </div>

                                <div>
                                    <motion.div
                                        className="text-5xl font-bold mb-2"
                                        initial={{ opacity: 0 }}
                                        whileInView={{ opacity: 1 }}
                                        transition={{ duration: 0.5, delay: 0.6 }}
                                        viewport={{ once: true }}
                                    >
                                        {setting ? setting?.projects_completion : "1000"}+
                                    </motion.div>
                                    <p className="text-yellow-300 text-sm uppercase tracking-wide font-medium">Projects Completed</p>
                                </div>
                            </div>
                        </div>
                    </motion.div>

                    <GalleryItems galleryImages={galleryImages} setSelectedImage={setSelectedImage} />

                    {/* Mission Statement */}
                    <motion.div
                        className="bg-white rounded-xl p-10 shadow-md border border-gray-100"
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.7, delay: 0.3 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl font-bold text-center text-gray-800 mb-6">Our Mission</h2>
                        <p className="text-lg text-center text-gray-600 max-w-3xl mx-auto">
                            {setting.mission ? setting?.mission : "To empower businesses through innovative solutions, exceptional service, and a commitment to excellence."}
                        </p>
                    </motion.div>
                </div>
            </section>

            {/* Image Modal (if selected) */}
            {selectedImage && (
                <div className="fixed inset-0 bg-black bg-opacity-90 flex items-center justify-center z-50 p-4" onClick={() => setSelectedImage(null)}>
                    <div className="max-w-4xl w-full">
                        <div className="bg-white rounded-xl overflow-hidden">
                            <div className="p-4 border-b border-gray-200 flex justify-between items-center">
                                <h3 className="text-xl font-bold text-gray-800">{selectedImage.title}</h3>
                                <button 
                                    className="text-gray-500 hover:text-gray-700"
                                    onClick={() => setSelectedImage(null)}
                                >
                                    <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                                    </svg>
                                </button>
                            </div>
                            <div className="p-4">
                                {/* Placeholder for larger image */}
                                <div className="w-full h-96 bg-gradient-to-r from-red-100 to-yellow-100 flex items-center justify-center">
                                    <span className="text-gray-500 text-xl font-medium">{selectedImage.title}</span>
                                </div>
                                <p className="mt-4 text-gray-600">Category: {selectedImage.category}</p>
                            </div>
                        </div>
                    </div>
                </div>
            )}

            <Footer setting={setting} />
        </div>
    );
};

export default About;