import React from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { 
    UserGroupIcon, 
    ClipboardDocumentCheckIcon, 
    AcademicCapIcon,
    StarIcon,
    PencilIcon,
    TrashIcon,
    PlusCircleIcon
} from '@heroicons/react/24/outline';

// --- Helper function to calculate analytics ---
const getAnalytics = (doctors) => {
    const totalDoctors = doctors.length;
    const totalBookedServices = doctors.reduce((sum, doctor) => sum + (doctor.booked_services_count || 0), 0);
    const totalSchoolServices = doctors.reduce((sum, doctor) => sum + (doctor.booked_school_services_count || 0), 0);
    
    const topPerformer = doctors.reduce((prev, current) => 
        (prev.booked_services_count + prev.booked_school_services_count) > (current.booked_services_count + current.booked_school_services_count) ? prev : current
    , doctors[0]);

    return { totalDoctors, totalBookedServices, totalSchoolServices, topPerformer };
};

// --- Sub-Component: Analytics Card ---
const AnalyticsCard = ({ title, value, icon, color }) => {
    return (
        <div className="bg-white overflow-hidden shadow-lg rounded-lg border border-gray-100">
            <div className="p-6">
                <div className="flex items-center">
                    <div className={`flex-shrink-0 p-3 rounded-lg ${color}`}>
                        {icon}
                    </div>
                    <div className="ml-5 w-0 flex-1">
                        <dl>
                            <dt className="text-sm font-medium text-gray-500 truncate uppercase tracking-wide">{title}</dt>
                            <dd className="text-2xl font-bold text-gray-900">{value}</dd>
                        </dl>
                    </div>
                </div>
            </div>
        </div>
    );
};

// --- Sub-Component: Doctor Card ---
const DoctorCard = ({ doctor }) => {
    return (
        <li className="bg-white shadow-sm rounded-lg overflow-hidden hover:shadow-md transition-shadow duration-300">
            <div className="p-6">
                <div className="flex items-center justify-between">
                    <div className="flex-1 min-w-0">
                        {/* Doctor Name and Contact Info */}
                        <p className="text-lg font-semibold text-gray-900 truncate">{doctor.name}</p>
                        <div className="mt-2 flex items-center text-sm text-gray-500">
                            <span className="truncate">{doctor.email}</span>
                        </div>
                        <p className="mt-1 text-sm text-gray-500">{doctor.mobile_number}</p>
                    </div>
                    
                    <div className="flex items-center space-x-6 ml-4">
                        {/* Service Counts */}
                        <div className="text-right">
                            <div className="flex items-center text-sm text-gray-600">
                                <AcademicCapIcon className="h-5 w-5 mr-2 text-emerald-500" />
                                <span className="font-medium">{doctor.booked_school_services_count}</span>
                                <span className="ml-1">School</span>
                            </div>
                            <div className="flex items-center text-sm text-gray-600 mt-1">
                                <ClipboardDocumentCheckIcon className="h-5 w-5 mr-2 text-emerald-500" />
                                <span className="font-medium">{doctor.booked_services_count}</span>
                                <span className="ml-1">Service</span>
                            </div>
                        </div>
                        
                        {/* Action Buttons */}
                        <div className="flex-shrink-0 flex space-x-2">
                            <Link
                                href={route('admin.doctors.edit', doctor.id)}
                                className="inline-flex items-center p-2 border border-gray-300 rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500"
                            >
                                <PencilIcon className="h-4 w-4" aria-hidden="true" />
                                <span className="sr-only">Edit {doctor.name}</span>
                            </Link>
                            <button
                                onClick={() => { /* Implement delete functionality */ }}
                                className="inline-flex items-center p-2 border border-transparent rounded-md text-red-700 bg-red-100 hover:bg-red-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                            >
                                <TrashIcon className="h-4 w-4" aria-hidden="true" />
                                <span className="sr-only">Delete {doctor.name}</span>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </li>
    );
};

export default function Index({ auth, doctors }) {
    const analytics = getAnalytics(doctors.data);

    return (
        <AdminLayout
            user={auth.user}
            header={
                <div>
                    <h2 className="font-semibold text-2xl text-gray-800 leading-tight">Doctors Dashboard</h2>
                    <p className="mt-1 text-sm text-gray-600">Manage your medical staff and track their performance.</p>
                </div>
            }
        >
            <Head title="Doctors" />

            <div className="py-6">
                <div className="mx-auto sm:px-6 lg:px-8">
                    {/* --- Analytics Section --- */}
                    <div className="grid grid-cols-1 gap-5 sm:grid-cols-2 lg:grid-cols-4 mb-8">
                        <AnalyticsCard
                            title="Total Doctors"
                            value={analytics.totalDoctors}
                            icon={<UserGroupIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-gradient-to-r from-emerald-500 to-emerald-600"
                        />
                        <AnalyticsCard
                            title="Total Booked Services"
                            value={analytics.totalBookedServices}
                            icon={<ClipboardDocumentCheckIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-gradient-to-r from-blue-500 to-blue-600"
                        />
                        <AnalyticsCard
                            title="Total School Services"
                            value={analytics.totalSchoolServices}
                            icon={<AcademicCapIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-gradient-to-r from-indigo-500 to-indigo-600"
                        />
                        <AnalyticsCard
                            title="Top Performer"
                            value={analytics.topPerformer?.name || 'N/A'}
                            icon={<StarIcon className="h-6 w-6 text-white" aria-hidden="true" />}
                            color="bg-gradient-to-r from-yellow-500 to-yellow-600"
                        />
                    </div>

                    {/* Header with Create Button */}
                    <div className="flex justify-between items-center mb-6">
                        <h3 className="text-2xl font-bold text-gray-900">All Doctors</h3>
                        <Link
                            href={route('admin.doctors.create')}
                            className="inline-flex items-center px-4 py-2 bg-emerald-600 border border-transparent rounded-md font-semibold text-sm text-white hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500 transition-all duration-200 shadow-sm"
                        >
                            <PlusCircleIcon className="-ml-1 mr-2 h-5 w-5" />
                            Add New Doctor
                        </Link>
                    </div>

                    {/* Doctors List */}
                    <div className="bg-white shadow-lg rounded-lg overflow-hidden">
                        {doctors.data.length > 0 ? (
                            <ul className="divide-y divide-gray-200">
                                {doctors.data.map((doctor) => (
                                    <DoctorCard key={doctor.id} doctor={doctor} />
                                ))}
                            </ul>
                        ) : (
                            // Beautiful Empty State
                            <div className="text-center py-12">
                                <UserGroupIcon className="mx-auto h-12 w-12 text-gray-400" />
                                <h3 className="mt-2 text-sm font-medium text-gray-900">No doctors</h3>
                                <p className="mt-1 text-sm text-gray-500">Get started by adding a new doctor to the system.</p>
                                <div className="mt-6">
                                    <Link
                                        href={route('admin.doctors.create')}
                                        className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-emerald-600 hover:bg-emerald-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500"
                                    >
                                        <PlusCircleIcon className="-ml-1 mr-2 h-5 w-5" aria-hidden="true" />
                                        New Doctor
                                    </Link>
                                </div>
                            </div>
                        )}
                    </div>

                    {/* --- Pagination --- */}
                    {doctors.links && doctors.links.length > 3 && (
                        <div className="mt-6 flex justify-center">
                            {doctors.links.map((link, key) => (
                                <Link
                                    key={key}
                                    href={link.url || '#'}
                                    className={`mr-1 mb-1 px-4 py-3 text-sm leading-4 border rounded-md transition-colors duration-150 ${
                                        link.active
                                            ? 'bg-emerald-600 text-white border-emerald-600'
                                            : 'bg-white text-gray-700 border-gray-300 hover:bg-gray-50'
                                    } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                />
                            ))}
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}