import React, { useState } from 'react';
import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link, useForm } from '@inertiajs/react';
import { formatDistanceToNow } from 'date-fns';
import { 
    CheckCircleIcon, 
    TrashIcon, 
    BellIcon, 
    CheckIcon,
    ExclamationTriangleIcon,
    UserIcon,
    ShoppingBagIcon
} from '@heroicons/react/24/outline';

// --- Helper function to get notification icon ---
const getNotificationIcon = (type) => {
    const icons = {
        'App\\Notifications\\NewOrder': <ShoppingBagIcon className="h-6 w-6 text-blue-500" />,
        'App\\Notifications\\NewUser': <UserIcon className="h-6 w-6 text-emerald-500" />,
        'App\\Notifications\\SystemAlert': <ExclamationTriangleIcon className="h-6 w-6 text-yellow-500" />,
    };
    return icons[type] || <BellIcon className="h-6 w-5 text-gray-400" />;
};

// --- Sub-Component: Notification Item ---
const NotificationItem = ({ notification, onMarkAsRead, onDelete }) => {
    const isUnread = !notification.read_at;

    return (
        <div
            className={`group flex items-start p-4 rounded-lg border transition-all duration-200 ${
                isUnread 
                    ? 'bg-emerald-50 border-emerald-200 shadow-sm' 
                    : 'bg-white border-gray-200 hover:bg-gray-50'
            }`}
        >
            {/* Icon & Indicator */}
            <div className="flex-shrink-0 mr-4">
                {getNotificationIcon(notification.type)}
                {isUnread && (
                    <span className="absolute -mt-5 -ml-1 h-3 w-3 bg-emerald-500 rounded-full animate-pulse"></span>
                )}
            </div>

            {/* Content */}
            <div className="flex-1 min-w-0">
                <p className={`text-sm ${isUnread ? 'font-semibold text-gray-900' : 'text-gray-700'}`}>
                    {notification.data.message}
                </p>
                <p className="text-xs text-gray-500 mt-1">
                    {formatDistanceToNow(new Date(notification.created_at), { addSuffix: true })}
                </p>
            </div>

            {/* Actions */}
            <div className="flex-shrink-0 ml-4 opacity-0 group-hover:opacity-100 transition-opacity duration-200">
                {isUnread && (
                    <button
                        onClick={() => onMarkAsRead(notification.id)}
                        className="inline-flex items-center p-1.5 rounded-full text-emerald-600 hover:bg-emerald-100 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-emerald-500 mr-2"
                        title="Mark as Read"
                    >
                        <CheckIcon className="h-4 w-4" aria-hidden="true" />
                    </button>
                )}
                <button
                    onClick={() => onDelete(notification.id)}
                    className="inline-flex items-center p-1.5 rounded-full text-red-600 hover:bg-red-100 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                    title="Delete"
                >
                    <TrashIcon className="h-4 w-4" aria-hidden="true" />
                </button>
            </div>
        </div>
    );
};

export default function Index({ auth, notifications }) {
    const { post, delete: destroy } = useForm();
    const [selectedNotifications, setSelectedNotifications] = useState([]);

    const handleSelectAll = (e) => {
        if (e.target.checked) {
            setSelectedNotifications(notifications.data.map(n => n.id));
        } else {
            setSelectedNotifications([]);
        }
    };

    const handleSelectOne = (id) => {
        setSelectedNotifications(prev =>
            prev.includes(id) ? prev.filter(nId => nId !== id) : [...prev, id]
        );
    };

    const markAsRead = (id) => {
        post(route('admin.notifications.markAsRead', id), { preserveScroll: true });
    };

    const markAllAsRead = () => {
        // You would need a backend route for this
        // post(route('admin.notifications.markAllAsRead'), { preserveScroll: true });
        alert('Mark all as read functionality would be implemented here.');
    };

    const deleteNotification = (id) => {
        if (confirm('Are you sure you want to delete this notification?')) {
            destroy(route('admin.notifications.destroy', id), { preserveScroll: true });
        }
    };

    const deleteSelected = () => {
        if (confirm(`Are you sure you want to delete ${selectedNotifications.length} selected notifications?`)) {
            // You would need a backend route for this
            // destroy(route('admin.notifications.destroySelected'), { data: { ids: selectedNotifications } }, { preserveScroll: true });
            alert('Delete selected functionality would be implemented here.');
        }
    };

    return (
        <AdminLayout
            user={auth.user}
            header={
                <div className="flex items-center justify-between">
                    <h2 className="font-semibold text-xl text-gray-800 leading-tight">Notifications</h2>
                    <span className="text-sm text-gray-500">
                        {notifications.data.filter(n => !n.read_at).length} unread
                    </span>
                </div>
            }
        >
            <Head title="Notifications" />

            <div className="py-6">
                <div className="max-w-4xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white shadow-xl rounded-lg overflow-hidden">
                        {/* Bulk Actions Header */}
                        {notifications.data.length > 0 && (
                            <div className="px-6 py-3 border-b border-gray-200 bg-gray-50 flex items-center justify-between">
                                <div className="flex items-center">
                                    <input
                                        type="checkbox"
                                        className="h-4 w-4 text-emerald-600 focus:ring-emerald-500 border-gray-300 rounded"
                                        onChange={handleSelectAll}
                                        checked={selectedNotifications.length === notifications.data.length && notifications.data.length > 0}
                                    />
                                    <label className="ml-2 text-sm text-gray-600">Select All</label>
                                </div>
                                <div className="flex space-x-2">
                                    {selectedNotifications.length > 0 && (
                                        <>
                                            <button
                                                onClick={markAllAsRead}
                                                className="text-xs font-medium text-emerald-700 hover:text-emerald-600"
                                            >
                                                Mark Selected as Read
                                            </button>
                                            <button
                                                onClick={deleteSelected}
                                                className="text-xs font-medium text-red-700 hover:text-red-600"
                                            >
                                                Delete Selected
                                            </button>
                                        </>
                                    )}
                                </div>
                            </div>
                        )}

                        <div className="p-6">
                            {notifications.data.length === 0 ? (
                                // Beautiful Empty State
                                <div className="text-center py-12">
                                    <BellIcon className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900">No notifications</h3>
                                    <p className="mt-1 text-sm text-gray-500">You're all caught up! Check back later for new updates.</p>
                                </div>
                            ) : (
                                <div className="space-y-3">
                                    {notifications.data.map((notification) => (
                                        <div key={notification.id} className="relative">
                                            <input
                                                type="checkbox"
                                                className="absolute top-4 left-4 h-4 w-4 text-emerald-600 focus:ring-emerald-500 border-gray-300 rounded"
                                                onChange={() => handleSelectOne(notification.id)}
                                                checked={selectedNotifications.includes(notification.id)}
                                            />
                                            <div className="ml-8">
                                                <NotificationItem
                                                    notification={notification}
                                                    onMarkAsRead={markAsRead}
                                                    onDelete={deleteNotification}
                                                />
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Pagination */}
                    {notifications.links && notifications.links.length > 3 && (
                        <div className="mt-6 flex justify-center">
                            {notifications.links.map((link, key) => (
                                <Link
                                    key={key}
                                    href={link.url || '#'}
                                    className={`mr-1 mb-1 px-4 py-3 text-sm leading-4 border rounded-md transition-colors duration-150 ${
                                        link.active
                                            ? 'bg-emerald-600 text-white border-emerald-600'
                                            : 'bg-white text-gray-700 border-gray-300 hover:bg-gray-50'
                                    } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                />
                            ))}
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}