import { Head, useForm } from '@inertiajs/react';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';
import AdminLayout from '@/Layouts/AdminLayout';

// Add a new Textarea component
const Textarea = ({ className = '', ...props }) => {
    return (
        <textarea
            className={`block w-full rounded-md border-gray-300 shadow-sm focus:border-indigo-500 focus:ring-indigo-500 ${className}`}
            {...props}
        />
    );
};

export default function CoreValues({ auth, setting }) {
    const { data, setData, post, errors, processing } = useForm({
        innovation: setting.innovation || '',
        teamwork: setting.teamwork || '',
        growth: setting.growth || '',
        excellence: setting.excellence || '',
        speed: setting.speed || '',
        integrity: setting.integrity || '',
        mission: setting.mission || '',
        vision: setting.vision || '',
    });

    const submit = (e) => {
        e.preventDefault();
        post(route('admin.settings.core-values.update'));
    };

    // Core values data with icons and descriptions
    const coreValues = [
        {
            id: 'innovation',
            label: 'Innovation',
            icon: (
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                </svg>
            ),
            description: 'Pioneering new ideas and solutions',
            color: 'bg-blue-100 text-blue-800'
        },
        {
            id: 'teamwork',
            label: 'Teamwork',
            icon: (
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                </svg>
            ),
            description: 'Collaborating for collective success',
            color: 'bg-green-100 text-green-800'
        },
        {
            id: 'growth',
            label: 'Growth',
            icon: (
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 7h8m0 0v8m0-8l-8 8-4-4-6 6" />
                </svg>
            ),
            description: 'Continuous development and improvement',
            color: 'bg-purple-100 text-purple-800'
        },
        {
            id: 'excellence',
            label: 'Excellence',
            icon: (
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M5 3v4M3 5h4M6 17v4m-2-2h4m5-16l2.286 6.857L21 12l-5.714 2.143L13 21l-2.286-6.857L5 12l5.714-2.143L13 3z" />
                </svg>
            ),
            description: 'Striving for the highest quality',
            color: 'bg-yellow-100 text-yellow-800'
        },
        {
            id: 'speed',
            label: 'Speed',
            icon: (
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M13 10V3L4 14h7v7l9-11h-7z" />
                </svg>
            ),
            description: 'Acting with agility and efficiency',
            color: 'bg-red-100 text-red-800'
        },
        {
            id: 'integrity',
            label: 'Integrity',
            icon: (
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                </svg>
            ),
            description: 'Upholding honesty and strong principles',
            color: 'bg-indigo-100 text-indigo-800'
        }
    ];

    return (
        <AdminLayout
            user={auth.user}
            header={
                <div className="flex flex-col md:flex-row md:items-center justify-between">
                    <h2 className="font-semibold text-xl text-gray-800 leading-tight">Core Values</h2>
                    <p className="mt-1 text-sm text-gray-600 md:mt-0">Manage your organization's core values</p>
                </div>
            }
        >
            <Head title="Core Values" />

            <div className="py-8">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white shadow-lg rounded-lg overflow-hidden">
                        {/* Header Section */}
                        <div className="bg-gradient-to-r from-blue-600 to-indigo-700 px-6 py-8">
                            <h3 className="text-2xl font-bold text-white">Organization Core Values</h3>
                            <p className="mt-2 text-blue-100">Define and update the core values that represent your organization's culture and principles</p>
                        </div>
                        
                        <div className="p-6">
                            <div className="mb-8">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0">
                                        <div className="flex items-center justify-center h-12 w-12 rounded-md bg-blue-100 text-blue-600">
                                            <svg className="h-6 w-6" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 21V5a2 2 0 00-2-2H7a2 2 0 00-2 2v16m14 0h2m-2 0h-5m-9 0H3m2 0h5M9 7h1m-1 4h1m4-4h1m-1 4h1m-5 10v-5a1 1 0 011-1h2a1 1 0 011 1v5m-4 0h4" />
                                            </svg>
                                        </div>
                                    </div>
                                    <div className="ml-4">
                                        <h4 className="text-lg font-medium text-gray-900">Core Values Configuration</h4>
                                        <p className="text-sm text-gray-500">Update the text for each of your organization's core values</p>
                                    </div>
                                </div>
                            </div>

                            <form onSubmit={submit}>
                                <div className="space-y-6 mb-8">
                                    <div>
                                        <InputLabel htmlFor="mission" value="Our Mission" />
                                        <Textarea
                                            id="mission"
                                            name="mission"
                                            value={data.mission}
                                            rows={5}
                                            className="mt-1 block w-full"
                                            placeholder="Describe your organization's mission..."
                                            onChange={(e) => setData('mission', e.target.value)}
                                        />
                                        <InputError message={errors.mission} className="mt-2" />
                                    </div>
                                    <div>
                                        <InputLabel htmlFor="vision" value="Our Vision" />
                                        <Textarea
                                            id="vision"
                                            name="vision"
                                            value={data.vision}
                                            rows={5}
                                            className="mt-1 block w-full"
                                            placeholder="Describe your organization's vision..."
                                            onChange={(e) => setData('vision', e.target.value)}
                                        />
                                        <InputError message={errors.vision} className="mt-2" />
                                    </div>
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                                    {coreValues.map((value) => (
                                        <div key={value.id} className="bg-white border border-gray-200 rounded-lg shadow-sm overflow-hidden hover:shadow-md transition-shadow duration-200">
                                            <div className={`px-4 py-3 ${value.color}`}>
                                                <div className="flex items-center">
                                                    <div className="flex-shrink-0">
                                                        {value.icon}
                                                    </div>
                                                    <div className="ml-3">
                                                        <h3 className="text-sm font-medium">{value.label}</h3>
                                                        <p className="text-xs opacity-80">{value.description}</p>
                                                    </div>
                                                </div>
                                            </div>
                                            <div className="p-4">
                                                <InputLabel htmlFor={value.id} className="sr-only" value={value.label} />
                                                <Textarea
                                                    id={value.id}
                                                    name={value.id}
                                                    value={data[value.id]}
                                                    rows={3}
                                                    className="mt-1 block w-full"
                                                    placeholder={`Describe your organization's commitment to ${value.label.toLowerCase()}...`}
                                                    onChange={(e) => setData(value.id, e.target.value)}
                                                />
                                                <InputError message={errors[value.id]} className="mt-2" />
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                <div className="mt-8 flex items-center justify-between">
                                    <div className="text-sm text-gray-500">
                                        <p>Last updated: {setting.updated_at || 'Never'}</p>
                                    </div>
                                    <div className="flex space-x-3">
                                        <button
                                            type="button"
                                            className="inline-flex items-center px-4 py-2 border border-gray-300 rounded-md shadow-sm text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                        >
                                            Cancel
                                        </button>
                                        <PrimaryButton className="ml-4" disabled={processing}>
                                            {processing ? 'Updating...' : 'Update Core Values'}
                                        </PrimaryButton>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}