import React, { useState } from 'react';
import { Head, Link } from '@inertiajs/react';
import HomeLayout from '@/Layouts/HomeLayout';
import { 
    CalendarIcon, 
    ClockIcon, 
    UserIcon, 
    TagIcon, 
    MagnifyingGlassIcon,
    HeartIcon,
    BookmarkIcon,
    ChatBubbleLeftIcon,
    ShareIcon,
    ArrowRightIcon,
    ChevronLeftIcon,
    ChevronRightIcon,
    EnvelopeIcon,
    BellIcon
} from '@heroicons/react/24/outline';
import { HeartIcon as HeartSolidIcon, BookmarkIcon as BookmarkSolidIcon } from '@heroicons/react/24/solid';

export default function Blog() {
    const [searchTerm, setSearchTerm] = useState('');
    const [selectedCategory, setSelectedCategory] = useState('all');
    const [likedPosts, setLikedPosts] = useState([]);
    const [savedPosts, setSavedPosts] = useState([]);
    const [email, setEmail] = useState('');
    const [currentPage, setCurrentPage] = useState(1);
    const postsPerPage = 6;

    // Mock data for blog posts
    const blogPosts = [
        {
            id: 1,
            title: "The Future of Web Development: Trends to Watch in 2023",
            excerpt: "Explore the cutting-edge technologies and frameworks that are shaping the future of web development, from AI integration to advanced CSS techniques.",
            image: "https://picsum.photos/seed/blog1/800/500.jpg",
            author: "Alex Johnson",
            date: "June 15, 2023",
            readTime: "5 min read",
            category: "Technology",
            tags: ["Web Dev", "JavaScript", "CSS"],
            likes: 42,
            comments: 8,
            featured: true
        },
        {
            id: 2,
            title: "Mastering Responsive Design: A Comprehensive Guide",
            excerpt: "Learn the principles and techniques for creating websites that look great on any device, from mobile phones to desktop computers.",
            image: "https://picsum.photos/seed/blog2/800/500.jpg",
            author: "Sarah Williams",
            date: "June 12, 2023",
            readTime: "8 min read",
            category: "Design",
            tags: ["Responsive", "CSS", "Mobile"],
            likes: 38,
            comments: 12,
            featured: true
        },
        {
            id: 3,
            title: "Building Scalable Applications with Microservices",
            excerpt: "Discover how microservices architecture can help you build more scalable, maintainable, and resilient applications.",
            image: "https://picsum.photos/seed/blog3/800/500.jpg",
            author: "Michael Chen",
            date: "June 8, 2023",
            readTime: "10 min read",
            category: "Architecture",
            tags: ["Microservices", "Backend", "Scalability"],
            likes: 56,
            comments: 15,
            featured: false
        },
        {
            id: 4,
            title: "The Art of User Experience: Creating Intuitive Interfaces",
            excerpt: "Dive into the principles of UX design and learn how to create interfaces that users love and find intuitive.",
            image: "https://picsum.photos/seed/blog4/800/500.jpg",
            author: "Emily Rodriguez",
            date: "June 5, 2023",
            readTime: "7 min read",
            category: "Design",
            tags: ["UX", "Interface", "Usability"],
            likes: 63,
            comments: 9,
            featured: false
        },
        {
            id: 5,
            title: "Securing Your Web Application: Best Practices",
            excerpt: "Learn the essential security measures every developer should implement to protect web applications from common vulnerabilities.",
            image: "https://picsum.photos/seed/blog5/800/500.jpg",
            author: "David Kim",
            date: "June 1, 2023",
            readTime: "12 min read",
            category: "Security",
            tags: ["Security", "Web Dev", "Best Practices"],
            likes: 71,
            comments: 18,
            featured: false
        },
        {
            id: 6,
            title: "Introduction to Machine Learning for Developers",
            excerpt: "Get started with machine learning and discover how it can enhance your applications with intelligent features.",
            image: "https://picsum.photos/seed/blog6/800/500.jpg",
            author: "Jessica Taylor",
            date: "May 28, 2023",
            readTime: "9 min read",
            category: "Technology",
            tags: ["Machine Learning", "AI", "Python"],
            likes: 85,
            comments: 22,
            featured: false
        },
        {
            id: 7,
            title: "Optimizing Database Performance: Tips and Tricks",
            excerpt: "Improve your database performance with these proven techniques for query optimization and efficient data management.",
            image: "https://picsum.photos/seed/blog7/800/500.jpg",
            author: "Robert Martinez",
            date: "May 24, 2023",
            readTime: "6 min read",
            category: "Database",
            tags: ["Database", "Performance", "SQL"],
            likes: 47,
            comments: 11,
            featured: false
        },
        {
            id: 8,
            title: "The Rise of Progressive Web Apps",
            excerpt: "Explore how Progressive Web Apps are bridging the gap between web and native applications, delivering better user experiences.",
            image: "https://picsum.photos/seed/blog8/800/500.jpg",
            author: "Amanda White",
            date: "May 20, 2023",
            readTime: "7 min read",
            category: "Technology",
            tags: ["PWA", "JavaScript", "Mobile"],
            likes: 52,
            comments: 14,
            featured: false
        },
        {
            id: 9,
            title: "Effective Team Collaboration in Remote Development",
            excerpt: "Discover tools and strategies for fostering effective collaboration among remote development teams.",
            image: "https://picsum.photos/seed/blog9/800/500.jpg",
            author: "Thomas Anderson",
            date: "May 16, 2023",
            readTime: "5 min read",
            category: "Management",
            tags: ["Remote Work", "Collaboration", "Teams"],
            likes: 39,
            comments: 7,
            featured: false
        }
    ];

    const categories = [
        { id: 'all', name: 'All Posts', count: blogPosts.length },
        { id: 'Technology', name: 'Technology', count: blogPosts.filter(post => post.category === 'Technology').length },
        { id: 'Design', name: 'Design', count: blogPosts.filter(post => post.category === 'Design').length },
        { id: 'Architecture', name: 'Architecture', count: blogPosts.filter(post => post.category === 'Architecture').length },
        { id: 'Security', name: 'Security', count: blogPosts.filter(post => post.category === 'Security').length },
        { id: 'Database', name: 'Database', count: blogPosts.filter(post => post.category === 'Database').length },
        { id: 'Management', name: 'Management', count: blogPosts.filter(post => post.category === 'Management').length }
    ];

    const popularTags = [
        { name: 'JavaScript', count: 24 },
        { name: 'CSS', count: 18 },
        { name: 'React', count: 21 },
        { name: 'Node.js', count: 15 },
        { name: 'Python', count: 19 },
        { name: 'Machine Learning', count: 12 },
        { name: 'UX Design', count: 16 },
        { name: 'Database', count: 14 }
    ];

    const toggleLike = (postId) => {
        setLikedPosts(prev => 
            prev.includes(postId) 
                ? prev.filter(id => id !== postId)
                : [...prev, postId]
        );
    };

    const toggleSave = (postId) => {
        setSavedPosts(prev => 
            prev.includes(postId) 
                ? prev.filter(id => id !== postId)
                : [...prev, postId]
        );
    };

    const handleSubscribe = (e) => {
        e.preventDefault();
        // Handle newsletter subscription
        alert(`Subscribed with email: ${email}`);
        setEmail('');
    };

    // Filter posts based on search term and selected category
    const filteredPosts = blogPosts.filter(post => {
        const matchesSearch = post.title.toLowerCase().includes(searchTerm.toLowerCase()) || 
                            post.excerpt.toLowerCase().includes(searchTerm.toLowerCase()) ||
                            post.tags.some(tag => tag.toLowerCase().includes(searchTerm.toLowerCase()));
        
        const matchesCategory = selectedCategory === 'all' || post.category === selectedCategory;
        
        return matchesSearch && matchesCategory;
    });

    // Pagination
    const indexOfLastPost = currentPage * postsPerPage;
    const indexOfFirstPost = indexOfLastPost - postsPerPage;
    const currentPosts = filteredPosts.slice(indexOfFirstPost, indexOfLastPost);
    const totalPages = Math.ceil(filteredPosts.length / postsPerPage);

    const paginate = (pageNumber) => setCurrentPage(pageNumber);

    return (
        <HomeLayout>
            <Head title="Blog" />
            
            {/* Hero Section */}
            <div className="relative bg-gradient-to-br from-green-900 via-green-800 to-emerald-900 text-white overflow-hidden">
                <div className="absolute inset-0 bg-black opacity-40"></div>
                <div className="relative container mx-auto px-4 py-24 md:py-32">
                    <div className="max-w-3xl mx-auto text-center">
                        <h1 className="text-4xl md:text-6xl font-bold mb-6">Our Blog</h1>
                        <p className="text-xl md:text-2xl mb-8 text-green-100">
                            Discover insights, tutorials, and industry news from our expert team
                        </p>
                        
                        {/* Search Bar */}
                        <div className="relative max-w-xl mx-auto">
                            <input
                                type="text"
                                placeholder="Search articles..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                className="w-full py-4 px-6 pr-12 rounded-full text-gray-800 bg-white shadow-lg focus:outline-none focus:ring-4 focus:ring-green-300"
                            />
                            <button className="absolute right-3 top-1/2 transform -translate-y-1/2 bg-green-700 text-white p-2 rounded-full hover:bg-green-800 transition-colors">
                                <MagnifyingGlassIcon className="h-5 w-5" />
                            </button>
                        </div>
                    </div>
                </div>
                
                {/* Decorative Elements */}
                <div className="absolute top-10 left-10 w-20 h-20 bg-emerald-600 rounded-full opacity-20 animate-pulse"></div>
                <div className="absolute bottom-20 right-20 w-32 h-32 bg-green-700 rounded-full opacity-20 animate-pulse"></div>
                <div className="absolute top-1/2 left-1/4 w-16 h-16 bg-teal-600 rounded-full opacity-20 animate-pulse"></div>
            </div>

            {/* Featured Posts */}
            <div className="container mx-auto px-4 py-16">
                <h2 className="text-3xl font-bold text-gray-800 mb-8">Featured Articles</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-8 mb-16">
                    {blogPosts.filter(post => post.featured).map(post => (
                        <div key={post.id} className="bg-white rounded-xl shadow-lg overflow-hidden transform transition-all duration-300 hover:scale-105 hover:shadow-xl">
                            <div className="relative h-64 overflow-hidden">
                                <img src={post.image} alt={post.title} className="w-full h-full object-cover" />
                                <div className="absolute top-4 left-4 bg-green-700 text-white px-3 py-1 rounded-full text-sm font-semibold">
                                    Featured
                                </div>
                            </div>
                            <div className="p-6">
                                <div className="flex items-center text-sm text-gray-500 mb-2">
                                    <span className="bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs font-medium mr-2">
                                        {post.category}
                                    </span>
                                    <span className="flex items-center mr-4">
                                        <CalendarIcon className="h-4 w-4 mr-1" />
                                        {post.date}
                                    </span>
                                    <span className="flex items-center">
                                        <ClockIcon className="h-4 w-4 mr-1" />
                                        {post.readTime}
                                    </span>
                                </div>
                                <h3 className="text-xl font-bold text-gray-800 mb-2">{post.title}</h3>
                                <p className="text-gray-600 mb-4">{post.excerpt}</p>
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center">
                                        <div className="w-10 h-10 bg-gray-300 rounded-full mr-3"></div>
                                        <span className="text-sm font-medium text-gray-700">{post.author}</span>
                                    </div>
                                    <Link href={`/blog/${post.id}`} className="text-green-700 hover:text-green-900 font-medium flex items-center">
                                        Read More
                                        <ArrowRightIcon className="h-4 w-4 ml-1" />
                                    </Link>
                                </div>
                            </div>
                        </div>
                    ))}
                </div>

                {/* Categories and Recent Posts */}
                <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
                    {/* Main Content */}
                    <div className="lg:col-span-3">
                        <h2 className="text-3xl font-bold text-gray-800 mb-8">Latest Articles</h2>
                        
                        {/* Blog Posts Grid */}
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-8">
                            {currentPosts.map(post => (
                                <div key={post.id} className="bg-white rounded-xl shadow-lg overflow-hidden transform transition-all duration-300 hover:scale-105 hover:shadow-xl">
                                    <div className="relative h-48 overflow-hidden">
                                        <img src={post.image} alt={post.title} className="w-full h-full object-cover" />
                                        <div className="absolute top-4 right-4 flex space-x-2">
                                            <button 
                                                onClick={() => toggleLike(post.id)}
                                                className="bg-white p-2 rounded-full shadow-md hover:bg-gray-100 transition-colors"
                                            >
                                                {likedPosts.includes(post.id) ? (
                                                    <HeartSolidIcon className="h-5 w-5 text-red-500" />
                                                ) : (
                                                    <HeartIcon className="h-5 w-5 text-gray-600" />
                                                )}
                                            </button>
                                            <button 
                                                onClick={() => toggleSave(post.id)}
                                                className="bg-white p-2 rounded-full shadow-md hover:bg-gray-100 transition-colors"
                                            >
                                                {savedPosts.includes(post.id) ? (
                                                    <BookmarkSolidIcon className="h-5 w-5 text-green-700" />
                                                ) : (
                                                    <BookmarkIcon className="h-5 w-5 text-gray-600" />
                                                )}
                                            </button>
                                        </div>
                                    </div>
                                    <div className="p-6">
                                        <div className="flex items-center text-sm text-gray-500 mb-2">
                                            <span className="bg-green-100 text-green-800 px-2 py-1 rounded-full text-xs font-medium mr-2">
                                                {post.category}
                                            </span>
                                            <span className="flex items-center mr-4">
                                                <CalendarIcon className="h-4 w-4 mr-1" />
                                                {post.date}
                                            </span>
                                            <span className="flex items-center">
                                                <ClockIcon className="h-4 w-4 mr-1" />
                                                {post.readTime}
                                            </span>
                                        </div>
                                        <h3 className="text-xl font-bold text-gray-800 mb-2">{post.title}</h3>
                                        <p className="text-gray-600 mb-4">{post.excerpt}</p>
                                        <div className="flex items-center justify-between">
                                            <div className="flex items-center">
                                                <div className="w-8 h-8 bg-gray-300 rounded-full mr-2"></div>
                                                <span className="text-sm font-medium text-gray-700">{post.author}</span>
                                            </div>
                                            <div className="flex items-center space-x-4 text-gray-500">
                                                <span className="flex items-center text-sm">
                                                    <HeartIcon className="h-4 w-4 mr-1" />
                                                    {post.likes}
                                                </span>
                                                <span className="flex items-center text-sm">
                                                    <ChatBubbleLeftIcon className="h-4 w-4 mr-1" />
                                                    {post.comments}
                                                </span>
                                                <span className="flex items-center text-sm">
                                                    <ShareIcon className="h-4 w-4 mr-1" />
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>

                        {/* Pagination */}
                        <div className="flex justify-center mt-12">
                            <nav className="flex items-center space-x-2">
                                <button
                                    onClick={() => currentPage > 1 && paginate(currentPage - 1)}
                                    disabled={currentPage === 1}
                                    className={`p-2 rounded-md ${currentPage === 1 ? 'bg-gray-200 text-gray-400 cursor-not-allowed' : 'bg-white text-gray-700 hover:bg-gray-100'}`}
                                >
                                    <ChevronLeftIcon className="h-5 w-5" />
                                </button>
                                
                                {[...Array(totalPages)].map((_, i) => (
                                    <button
                                        key={i}
                                        onClick={() => paginate(i + 1)}
                                        className={`px-4 py-2 rounded-md ${currentPage === i + 1 ? 'bg-green-700 text-white' : 'bg-white text-gray-700 hover:bg-gray-100'}`}
                                    >
                                        {i + 1}
                                    </button>
                                ))}
                                
                                <button
                                    onClick={() => currentPage < totalPages && paginate(currentPage + 1)}
                                    disabled={currentPage === totalPages}
                                    className={`p-2 rounded-md ${currentPage === totalPages ? 'bg-gray-200 text-gray-400 cursor-not-allowed' : 'bg-white text-gray-700 hover:bg-gray-100'}`}
                                >
                                    <ChevronRightIcon className="h-5 w-5" />
                                </button>
                            </nav>
                        </div>
                    </div>

                    {/* Sidebar */}
                    <div className="lg:col-span-1">
                        {/* Categories */}
                        <div className="bg-white rounded-xl shadow-lg p-6 mb-8">
                            <h3 className="text-xl font-bold text-gray-800 mb-4">Categories</h3>
                            <ul className="space-y-2">
                                {categories.map(category => (
                                    <li key={category.id}>
                                        <button
                                            onClick={() => setSelectedCategory(category.id)}
                                            className={`w-full text-left px-4 py-2 rounded-lg transition-colors ${selectedCategory === category.id ? 'bg-green-100 text-green-800 font-medium' : 'hover:bg-gray-100 text-gray-700'}`}
                                        >
                                            <div className="flex justify-between items-center">
                                                <span>{category.name}</span>
                                                <span className="bg-gray-200 text-gray-600 text-xs px-2 py-1 rounded-full">
                                                    {category.count}
                                                </span>
                                            </div>
                                        </button>
                                    </li>
                                ))}
                            </ul>
                        </div>

                        {/* Popular Tags */}
                        <div className="bg-white rounded-xl shadow-lg p-6 mb-8">
                            <h3 className="text-xl font-bold text-gray-800 mb-4">Popular Tags</h3>
                            <div className="flex flex-wrap gap-2">
                                {popularTags.map((tag, index) => (
                                    <button
                                        key={index}
                                        onClick={() => setSearchTerm(tag.name)}
                                        className="bg-gray-100 text-gray-700 px-3 py-1 rounded-full text-sm hover:bg-green-100 hover:text-green-800 transition-colors"
                                    >
                                        #{tag.name} ({tag.count})
                                    </button>
                                ))}
                            </div>
                        </div>

                        {/* Newsletter */}
                        <div className="bg-gradient-to-br from-green-700 to-emerald-800 rounded-xl shadow-lg p-6 text-white">
                            <div className="flex items-center mb-4">
                                <EnvelopeIcon className="h-8 w-8 mr-3" />
                                <h3 className="text-xl font-bold">Subscribe to Our Newsletter</h3>
                            </div>
                            <p className="mb-4 text-green-100">
                                Get the latest articles and updates delivered straight to your inbox.
                            </p>
                            <form onSubmit={handleSubscribe}>
                                <input
                                    type="email"
                                    placeholder="Your email address"
                                    value={email}
                                    onChange={(e) => setEmail(e.target.value)}
                                    required
                                    className="w-full py-2 px-4 rounded-lg text-gray-800 mb-3 focus:outline-none focus:ring-2 focus:ring-white"
                                />
                                <button
                                    type="submit"
                                    className="w-full bg-white text-green-700 py-2 px-4 rounded-lg font-semibold hover:bg-green-50 transition-colors flex items-center justify-center"
                                >
                                    Subscribe
                                    <BellIcon className="h-5 w-5 ml-2" />
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>

            {/* Call to Action */}
            <div className="bg-gray-100 py-16">
                <div className="container mx-auto px-4 text-center">
                    <h2 className="text-3xl font-bold text-gray-800 mb-4">Enjoyed Our Articles?</h2>
                    <p className="text-xl text-gray-600 mb-8 max-w-2xl mx-auto">
                        Join our community of readers and stay updated with the latest insights and trends in technology and design.
                    </p>
                    <div className="flex flex-col sm:flex-row gap-4 justify-center">
                        <Link href="/register" className="bg-green-700 hover:bg-green-800 text-white font-bold py-3 px-8 rounded-lg transition-colors">
                            Create Account
                        </Link>
                        <Link href="/contact" className="bg-white hover:bg-gray-100 text-green-700 font-bold py-3 px-8 rounded-lg border-2 border-green-700 transition-colors">
                            Contact Us
                        </Link>
                    </div>
                </div>
            </div>
        </HomeLayout>
    );
}