import React, { useState, useEffect } from 'react';
import { Head, Link } from '@inertiajs/react';
import HomeLayout from '@/Layouts/HomeLayout';
import { motion } from 'framer-motion';

export default function Index({ posts }) {
    const [searchTerm, setSearchTerm] = useState('');
    const [selectedCategory, setSelectedCategory] = useState('all');
    const [readingTime, setReadingTime] = useState({});
    const [bookmarkedPosts, setBookmarkedPosts] = useState([]);
    const [scrollProgress, setScrollProgress] = useState(0);

    // Calculate reading progress
    useEffect(() => {
        const handleScroll = () => {
            const scrollTop = window.scrollY;
            const docHeight = document.documentElement.scrollHeight - window.innerHeight;
            const scrollPercent = (scrollTop / docHeight) * 100;
            setScrollProgress(scrollPercent);
        };

        window.addEventListener('scroll', handleScroll);
        return () => window.removeEventListener('scroll', handleScroll);
    }, []);

    // Calculate reading time for posts
    useEffect(() => {
        const calculateReadingTime = () => {
            const times = {};
            posts.data.forEach(post => {
                const wordsPerMinute = 200;
                const wordCount = post.content.split(' ').length;
                times[post.id] = Math.ceil(wordCount / wordsPerMinute);
            });
            setReadingTime(times);
        };
        calculateReadingTime();
    }, [posts]);

    // Format date function
    const formatDate = (dateString) => {
        const options = { year: 'numeric', month: 'long', day: 'numeric' };
        return new Date(dateString).toLocaleDateString(undefined, options);
    };


    // Toggle bookmark
    const toggleBookmark = (postId) => {
        setBookmarkedPosts(prev => 
            prev.includes(postId) 
                ? prev.filter(id => id !== postId)
                : [...prev, postId]
        );
    };

    // Filter posts based on search and category
    const filteredPosts = posts.data.filter(post => {
        const matchesSearch = post.title.toLowerCase().includes(searchTerm.toLowerCase()) ||
                             post.content.toLowerCase().includes(searchTerm.toLowerCase());
        const matchesCategory = selectedCategory === 'all' || post.category === selectedCategory;
        return matchesSearch && matchesCategory;
    });

    // Animation variants
    const containerVariants = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.1,
                delayChildren: 0.2,
            },
        },
    };

    const itemVariants = {
        hidden: { y: 30, opacity: 0 },
        visible: {
            y: 0,
            opacity: 1,
            transition: { duration: 0.6, ease: "easeOut" },
        },
    };

    return (
        <HomeLayout>
            <Head title="Blog" />

            {/* Reading Progress Bar */}
            <div className="fixed top-0 left-0 w-full h-1 bg-gray-200 z-50">
                <motion.div 
                    className="h-full bg-gradient-to-r from-emerald-500 to-teal-600"
                    style={{ width: `${scrollProgress}%` }}
                />
            </div>

            {/* Hero Section with Animated Background */}
            <div className="relative bg-gradient-to-br from-emerald-500 via-teal-600 to-cyan-700 text-white overflow-hidden">
                <div className="absolute inset-0">
                    <div className="absolute inset-0 bg-black opacity-20"></div>
                    <motion.div 
                        className="absolute -top-40 -right-40 w-80 h-80 bg-white rounded-full opacity-10 blur-3xl"
                        animate={{ 
                            x: [0, 100, 0],
                            y: [0, -100, 0],
                        }}
                        transition={{ 
                            duration: 20,
                            repeat: Infinity,
                            repeatType: "reverse"
                        }}
                    />
                    <motion.div 
                        className="absolute -bottom-40 -left-40 w-80 h-80 bg-white rounded-full opacity-10 blur-3xl"
                        animate={{ 
                            x: [0, -100, 0],
                            y: [0, 100, 0],
                        }}
                        transition={{ 
                            duration: 15,
                            repeat: Infinity,
                            repeatType: "reverse"
                        }}
                    />
                </div>
                
                <div className="relative max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-24 md:py-32">
                    <motion.div 
                        className="text-center"
                        initial={{ opacity: 0, y: 20 }}
                        animate={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.8 }}
                    >
                        <h1 className="text-4xl md:text-6xl font-bold mb-6 bg-clip-text text-transparent bg-gradient-to-r from-white to-cyan-100">
                            Our Blog
                        </h1>
                        <p className="text-xl md:text-2xl max-w-3xl mx-auto opacity-90 leading-relaxed">
                            Discover insights, tips, and stories from our team of experts
                        </p>
                    </motion.div>
                </div>
                
                <motion.div 
                    className="absolute bottom-0 left-0 right-0 h-20 bg-gradient-to-t from-white to-transparent"
                    initial={{ opacity: 0 }}
                    animate={{ opacity: 1 }}
                    transition={{ delay: 0.5, duration: 1 }}
                />
            </div>

            {/* Search and Filter Section */}
            <section className="py-8 bg-white shadow-sm sticky top-0 z-40">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row justify-between items-center space-y-4 md:space-y-0">
                        <div className="relative w-full md:w-96">
                            <input
                                type="text"
                                placeholder="Search articles..."
                                value={searchTerm}
                                onChange={(e) => setSearchTerm(e.target.value)}
                                className="w-full pl-12 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-emerald-500 focus:border-emerald-500"
                            />
                            <svg className="absolute left-4 top-3.5 w-5 h-5 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z" />
                            </svg>
                        </div>
                        
                    </div>
                </div>
            </section>

            {/* Featured Post */}
            {filteredPosts.length > 0 && (
                <section className="py-16 bg-gradient-to-b from-white to-gray-50">
                    <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                        <motion.div 
                            className="text-center mb-12"
                            initial={{ opacity: 0 }}
                            whileInView={{ opacity: 1 }}
                            viewport={{ once: true }}
                        >
                            <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">Featured Article</h2>
                            <div className="w-24 h-1 bg-gradient-to-r from-emerald-500 to-teal-600 mx-auto"></div>
                        </motion.div>
                        
                        <motion.div 
                            className="bg-white rounded-3xl shadow-2xl overflow-hidden group"
                            initial={{ opacity: 0, y: 30 }}
                            whileInView={{ opacity: 1, y: 0 }}
                            viewport={{ once: true }}
                            transition={{ duration: 0.8 }}
                        >
                            <div className="md:flex">
                                <div className="md:w-1/2 relative overflow-hidden">
                                    <img 
                                        src={filteredPosts[0].image_url} 
                                        alt={filteredPosts[0].title} 
                                        className="w-full h-64 md:h-full object-cover group-hover:scale-105 transition-transform duration-700"
                                    />
                                    <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent"></div>
                                    <div className="absolute bottom-6 left-6 text-white">
                                        <span className="text-sm font-medium opacity-90">Featured</span>
                                        <h3 className="text-2xl font-bold mt-1">{filteredPosts[0].title}</h3>
                                    </div>
                                </div>
                                <div className="md:w-1/2 p-8 lg:p-12">
                                    <div className="flex items-center text-sm text-gray-500 mb-6">
                                        <div className="flex items-center mr-6">
                                            <div className="w-10 h-10 bg-gradient-to-br from-emerald-400 to-teal-600 rounded-full mr-3 flex items-center justify-center text-white font-bold">
                                                {(filteredPosts[0].author || 'Admin')[0].toUpperCase()}
                                            </div>
                                            <div>
                                                <div className="font-medium text-gray-900">{filteredPosts[0].author || 'Admin'}</div>
                                                <div className="text-xs">Senior Writer</div>
                                            </div>
                                        </div>
                                        <div className="flex items-center space-x-4">
                                            <span className="flex items-center">
                                                <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                                </svg>
                                                {formatDate(filteredPosts[0].created_at)}
                                            </span>
                                            <span className="flex items-center">
                                                <svg className="w-4 h-4 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                                                </svg>
                                                {readingTime[filteredPosts[0].id] || 5} min read
                                            </span>
                                        </div>
                                    </div>
                                    
                                    <p className="text-gray-600 mb-6 leading-relaxed line-clamp-4">
                                        {filteredPosts[0].content}
                                    </p>
                                    
                                    <div className="flex items-center justify-between">
                                        <Link 
                                            href={route('blog.show', filteredPosts[0].id)} 
                                            className="inline-flex items-center text-emerald-600 font-semibold hover:text-emerald-700 transition-colors group"
                                        >
                                            Read Full Article
                                            <svg className="w-5 h-5 ml-2 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14 5l7 7m0 0l-7 7m7-7H3" />
                                            </svg>
                                        </Link>
                                        
                                        <div className="flex items-center space-x-3">
                                            <button className="text-gray-400 hover:text-blue-500 transition-colors">
                                                <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8.684 13.342C8.886 12.938 9 12.482 9 12c0-.482-.114-.938-.316-1.342m0 2.684a3 3 0 110-2.684m9.032 4.026a9.001 9.001 0 01-7.432 0m9.032-4.026A9.001 9.001 0 0112 3c-4.474 0-8.268 3.12-9.032 7.326m0 0A9.001 9.001 0 0012 21c4.474 0 8.268-3.12 9.032-7.326" />
                                                </svg>
                                            </button>
                                            <span className="text-sm text-gray-500">234 shares</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </motion.div>
                    </div>
                </section>
            )}

            {/* Blog Posts Grid */}
            <section className="py-16 bg-gray-50">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <motion.div 
                        className="text-center mb-12"
                        initial={{ opacity: 0 }}
                        whileInView={{ opacity: 1 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">Latest Articles</h2>
                        <p className="text-lg text-gray-600">Explore our most recent blog posts</p>
                    </motion.div>

                    {filteredPosts.length > 1 ? (
                        <motion.div 
                            className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8"
                            variants={containerVariants}
                            initial="hidden"
                            whileInView="visible"
                            viewport={{ once: true }}
                        >
                            {filteredPosts.slice(1).map((post, index) => (
                                <motion.div 
                                    key={post.id} 
                                    className="bg-white rounded-2xl shadow-lg overflow-hidden hover:shadow-2xl transition-all duration-300 group"
                                    variants={itemVariants}
                                    whileHover={{ y: -5 }}
                                >
                                    <div className="relative h-56 overflow-hidden">
                                        <img 
                                            src={post.image_url} 
                                            alt={post.title} 
                                            className="w-full h-full object-cover group-hover:scale-110 transition-transform duration-700"
                                        />
                                        <div className="absolute inset-0 bg-gradient-to-t from-black/50 to-transparent opacity-0 group-hover:opacity-100 transition-opacity duration-300"></div>
                                        
                                        <div className="absolute top-4 right-4 opacity-0 group-hover:opacity-100 transition-opacity duration-300">
                                            <button
                                                onClick={() => toggleBookmark(post.id)}
                                                className="w-10 h-10 bg-white/90 backdrop-blur rounded-full flex items-center justify-center text-gray-600 hover:text-red-500 transition-colors"
                                            >
                                                <svg className={`w-5 h-5 ${bookmarkedPosts.includes(post.id) ? 'fill-current text-red-500' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                                                </svg>
                                            </button>
                                        </div>
                                    </div>
                                    
                                    <div className="p-6">
                                        <div className="flex items-center justify-between mb-4">
                                            <div className="flex items-center text-sm text-gray-500">
                                                <div className="w-8 h-8 bg-gradient-to-br from-emerald-400 to-teal-600 rounded-full mr-2 flex items-center justify-center text-white text-xs font-bold">
                                                    {(post.author || 'Admin')[0].toUpperCase()}
                                                </div>
                                                <span>{post.author || 'Admin'}</span>
                                            </div>
                                            <span className="text-xs text-gray-500">{readingTime[post.id] || 5} min</span>
                                        </div>
                                        
                                        <h3 className="text-xl font-bold text-gray-900 mb-3 group-hover:text-emerald-600 transition-colors line-clamp-2">
                                            {post.title}
                                        </h3>
                                        
                                        <p className="text-gray-600 mb-4 line-clamp-3 leading-relaxed">
                                            {post.content.substring(0, 120)}...
                                        </p>
                                        
                                        <div className="flex items-center justify-between">
                                            <Link 
                                                href={route('blog.show', post.id)} 
                                                className="inline-flex items-center text-emerald-600 font-medium hover:text-emerald-700 transition-colors"
                                            >
                                                Read More
                                                <svg className="w-4 h-4 ml-1 group-hover:translate-x-1 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14 5l7 7m0 0l-7 7m7-7H3" />
                                                </svg>
                                            </Link>
                                            
                                            <div className="flex items-center space-x-2 text-gray-400">
                                                <button className="hover:text-red-500 transition-colors">
                                                    <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4.318 6.318a4.5 4.5 0 000 6.364L12 20.364l7.682-7.682a4.5 4.5 0 00-6.364-6.364L12 7.636l-1.318-1.318a4.5 4.5 0 00-6.364 0z" />
                                                    </svg>
                                                </button>
                                                <span className="text-xs">42</span>
                                            </div>
                                        </div>
                                    </div>
                                </motion.div>
                            ))}
                        </motion.div>
                    ) : (
                        <div className="text-center py-12">
                            <svg className="w-16 h-16 mx-auto text-gray-400 mb-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.172 16.172a4 4 0 015.656 0M9 10h.01M15 10h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
                            </svg>
                            <p className="text-gray-500 text-lg">No articles found matching your criteria.</p>
                            <button 
                                onClick={() => { setSearchTerm(''); setSelectedCategory('all'); }}
                                className="mt-4 px-6 py-2 bg-emerald-600 text-white rounded-lg hover:bg-emerald-700 transition-colors"
                            >
                                Clear Filters
                            </button>
                        </div>
                    )}

                    {/* Pagination */}
                    {posts.links && (
                        <motion.div 
                            className="mt-16 flex justify-center"
                            initial={{ opacity: 0 }}
                            whileInView={{ opacity: 1 }}
                            viewport={{ once: true }}
                        >
                            <div className="flex space-x-2">
                                {posts.links.map((link, index) => (
                                    <Link
                                        key={index}
                                        href={link.url || '#'}
                                        className={`px-4 py-2 rounded-lg font-medium transition-all ${
                                            link.active
                                                ? 'bg-gradient-to-r from-emerald-500 to-teal-600 text-white shadow-lg'
                                                : 'bg-white text-gray-700 hover:bg-gray-100 border border-gray-300'
                                        } ${!link.url ? 'cursor-not-allowed opacity-50' : ''}`}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </div>
                        </motion.div>
                    )}
                </div>
            </section>

            {/* Newsletter Section with Enhanced Design */}
            <section className="py-20 bg-gradient-to-br from-emerald-500 via-teal-600 to-cyan-700 relative overflow-hidden">
                <div className="absolute inset-0">
                    <motion.div 
                        className="absolute top-10 left-10 w-32 h-32 bg-white/10 rounded-full"
                        animate={{ 
                            scale: [1, 1.2, 1],
                            opacity: [0.3, 0.6, 0.3],
                        }}
                        transition={{ 
                            duration: 4,
                            repeat: Infinity,
                            repeatType: "reverse"
                        }}
                    />
                    <motion.div 
                        className="absolute bottom-10 right-10 w-40 h-40 bg-white/10 rounded-full"
                        animate={{ 
                            scale: [1.2, 1, 1.2],
                            opacity: [0.6, 0.3, 0.6],
                        }}
                        transition={{ 
                            duration: 5,
                            repeat: Infinity,
                            repeatType: "reverse"
                        }}
                    />
                </div>
                
                <div className="relative max-w-4xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                    <motion.div
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        viewport={{ once: true }}
                        transition={{ duration: 0.8 }}
                    >
                        <h2 className="text-3xl md:text-4xl font-bold text-white mb-4">
                            Stay Updated with Our Newsletter
                        </h2>
                        <p className="text-xl text-white/90 mb-8 max-w-2xl mx-auto">
                            Get the latest insights, exclusive content, and special offers delivered directly to your inbox
                        </p>
                        
                        <form className="flex flex-col sm:flex-row max-w-md mx-auto gap-4">
                            <input
                                type="email"
                                placeholder="Enter your email address"
                                className="flex-1 px-6 py-4 rounded-lg focus:outline-none focus:ring-4 focus:ring-white/30 text-gray-900"
                                required
                            />
                            <motion.button
                                type="submit"
                                className="px-8 py-4 bg-gray-900 text-white font-semibold rounded-lg hover:bg-gray-800 transition-colors shadow-lg"
                                whileHover={{ scale: 1.05 }}
                                whileTap={{ scale: 0.95 }}
                            >
                                Subscribe
                            </motion.button>
                        </form>
                        
                        <p className="mt-4 text-sm text-white/70">
                            Join 5,000+ subscribers. No spam, unsubscribe anytime.
                        </p>
                    </motion.div>
                </div>
            </section>
        </HomeLayout>
    );
}