import React, { useEffect, useState } from 'react';
import { Head, useForm } from '@inertiajs/react';
import Navbar from '@/Components/Navbar';
import Footer from '@/Components/Footer';
import InputError from '@/Components/InputError';
import InputLabel from '@/Components/InputLabel';
import PrimaryButton from '@/Components/PrimaryButton';
import TextInput from '@/Components/TextInput';
import { FiUser, FiMail, FiPhone, FiBriefcase, FiMessageSquare, FiFileText, FiCheck, FiPhoneCall } from 'react-icons/fi';
import Preloader from '@/Components/Preloader';

export default function Quote({ setting }) {
    const { data, setData, post, processing, errors, reset } = useForm({
        name: '',
        email: '',
        phone: '',
        service: '',
        message: '',
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route('quote.store'), {
            onSuccess: () => {
                reset();
                alert('Your quote request has been sent successfully!');
            },
        });
    };

    const [loading, setLoading] = useState(true);
    
    useEffect(() => {
        const timer = setTimeout(() => {
            setLoading(false);
        }, 1000);

        return () => clearTimeout(timer);
    }, []);

    if (loading) {
        return <Preloader />;
    }

    return (
        <div className="min-h-screen bg-gray-50 font-sans antialiased mt-24">
            <Head title="Get a Quote" />
            <Navbar setting={setting} />

            {/* Professional Hero Section */}
            <div className="relative bg-gradient-to-r from-red-700 to-red-800 py-16 text-white">
                <div className="container mx-auto px-6">
                    <div className="max-w-3xl">
                        <h1 className="text-4xl md:text-5xl font-bold mb-4">Request a Quote</h1>
                        <p className="text-xl text-red-100 max-w-2xl">
                            Get a customized proposal for your project. Our team will respond within one business day with a detailed estimate.
                        </p>
                    </div>
                </div>
                
                {/* Subtle decorative elements */}
                <div className="absolute top-0 right-0 w-96 h-96 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
            </div>

            <div className="container mx-auto px-6 py-12">
                <div className="max-w-6xl mx-auto">
                    <div className="bg-white rounded-xl shadow-lg overflow-hidden border border-gray-200">
                        <div className="grid grid-cols-1 lg:grid-cols-3">
                            {/* Left side - Form */}
                            <div className="lg:col-span-2">
                                {/* Professional Form Header */}
                                <div className="bg-gray-50 border-b border-gray-200 py-6 px-8">
                                    <h2 className="text-2xl font-bold text-gray-800">Project Information</h2>
                                    <p className="text-gray-600 mt-1">Please provide details about your project requirements</p>
                                </div>
                                
                                <div className="p-8">
                                    <form onSubmit={handleSubmit} className="space-y-6">
                                        <div>
                                            <InputLabel htmlFor="name" value="Full Name" className="flex items-center text-gray-700 font-medium mb-2">
                                                <FiUser className="mr-2 text-red-600" /> Your Name
                                            </InputLabel>
                                            <TextInput
                                                id="name"
                                                type="text"
                                                name="name"
                                                value={data.name}
                                                className="mt-1 block w-full border-gray-300 focus:border-red-500 focus:ring-red-500 rounded-lg shadow-sm"
                                                placeholder="Enter your full name"
                                                isFocused={true}
                                                onChange={(e) => setData('name', e.target.value)}
                                            />
                                            <InputError message={errors.name} className="mt-2" />
                                        </div>

                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                            <div>
                                                <InputLabel htmlFor="email" value="Email Address" className="flex items-center text-gray-700 font-medium mb-2">
                                                    <FiMail className="mr-2 text-red-600" /> Email
                                                </InputLabel>
                                                <TextInput
                                                    id="email"
                                                    type="email"
                                                    name="email"
                                                    value={data.email}
                                                    className="mt-1 block w-full border-gray-300 focus:border-red-500 focus:ring-red-500 rounded-lg shadow-sm"
                                                    placeholder="your.email@company.com"
                                                    onChange={(e) => setData('email', e.target.value)}
                                                />
                                                <InputError message={errors.email} className="mt-2" />
                                            </div>

                                            <div>
                                                <InputLabel htmlFor="phone" value="Phone Number" className="flex items-center text-gray-700 font-medium mb-2">
                                                    <FiPhone className="mr-2 text-red-600" /> Phone
                                                </InputLabel>
                                                <TextInput
                                                    id="phone"
                                                    type="text"
                                                    name="phone"
                                                    value={data.phone}
                                                    className="mt-1 block w-full border-gray-300 focus:border-red-500 focus:ring-red-500 rounded-lg shadow-sm"
                                                    placeholder="+1 (555) 123-4567"
                                                    onChange={(e) => setData('phone', e.target.value)}
                                                />
                                                <InputError message={errors.phone} className="mt-2" />
                                            </div>
                                        </div>

                                        <div>
                                            <InputLabel htmlFor="service" value="Service Type" className="flex items-center text-gray-700 font-medium mb-2">
                                                <FiBriefcase className="mr-2 text-red-600" /> Service
                                            </InputLabel>
                                            <select
                                                id="service"
                                                name="service"
                                                value={data.service}
                                                className="mt-1 block w-full border-gray-300 focus:border-red-500 focus:ring-red-500 rounded-lg shadow-sm px-4 py-3"
                                                onChange={(e) => setData('service', e.target.value)}
                                            >
                                                <option value="">Select a service</option>
                                                <option value="Web Development">Web Development</option>
                                                <option value="Mobile App">Mobile App Development</option>
                                                <option value="UI/UX Design">UI/UX Design</option>
                                                <option value="Consulting">IT Consulting</option>
                                                <option value="Other">Other</option>
                                            </select>
                                            <InputError message={errors.service} className="mt-2" />
                                        </div>

                                        <div>
                                            <InputLabel htmlFor="message" value="Project Description" className="flex items-center text-gray-700 font-medium mb-2">
                                                <FiMessageSquare className="mr-2 text-red-600" /> Details
                                            </InputLabel>
                                            <textarea
                                                id="message"
                                                name="message"
                                                rows="5"
                                                value={data.message}
                                                className="mt-1 block w-full border-gray-300 focus:border-red-500 focus:ring-red-500 rounded-lg shadow-sm"
                                                placeholder="Describe your project requirements, goals, timeline, and budget considerations..."
                                                onChange={(e) => setData('message', e.target.value)}
                                            ></textarea>
                                            <InputError message={errors.message} className="mt-2" />
                                        </div>

                                        <div className="flex items-center justify-between pt-6 border-t border-gray-200">
                                            <div>
                                                <p className="text-sm text-gray-500">
                                                    <span className="font-medium text-green-600">✓</span> Secure and confidential
                                                </p>
                                                <p className="text-xs text-gray-400 mt-1">We'll respond within 1 business day</p>
                                            </div>
                                            <PrimaryButton 
                                                disabled={processing}
                                                className="bg-gradient-to-r from-red-600 to-red-700 hover:from-red-700 hover:to-red-800 px-8 py-3 font-medium rounded-lg shadow-md transition-all duration-300 text-white"
                                            >
                                                {processing ? 'Sending...' : 'Submit Request'}
                                            </PrimaryButton>
                                        </div>
                                    </form>
                                </div>
                            </div>

                            {/* Right side - Corporate Information */}
                            <div className="bg-gradient-to-b from-gray-50 to-gray-100 p-8 flex flex-col">
                                <div className="mb-8 flex justify-center">
                                    <div className="flex items-center justify-center w-24 h-24 rounded-full bg-white shadow-md border border-gray-200">
                                        <FiFileText className="w-12 h-12 text-red-600" />
                                    </div>
                                </div>
                                
                                <h3 className="text-xl font-bold text-gray-800 mb-4 text-center">Why Request a Quote?</h3>
                                
                                <div className="space-y-4 mb-8">
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0 mt-1">
                                            <FiCheck className="h-5 w-5 text-green-500" />
                                        </div>
                                        <p className="ml-3 text-gray-700">Custom pricing tailored to your specific requirements</p>
                                    </div>
                                    
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0 mt-1">
                                            <FiCheck className="h-5 w-5 text-green-500" />
                                        </div>
                                        <p className="ml-3 text-gray-700">Detailed project timeline and deliverables</p>
                                    </div>
                                    
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0 mt-1">
                                            <FiCheck className="h-5 w-5 text-green-500" />
                                        </div>
                                        <p className="ml-3 text-gray-700">No-obligation consultation with our experts</p>
                                    </div>
                                    
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0 mt-1">
                                            <FiCheck className="h-5 w-5 text-green-500" />
                                        </div>
                                        <p className="ml-3 text-gray-700">Priority response within 1 business day</p>
                                    </div>
                                </div>
                                
                                <div className="mt-auto bg-white rounded-lg p-4 border border-gray-200 shadow-sm">
                                    <div className="flex items-center mb-2">
                                        <FiPhoneCall className="h-5 w-5 text-red-600 mr-2" />
                                        <h4 className="font-medium text-gray-800">Need immediate assistance?</h4>
                                    </div>
                                    <p className="text-sm text-gray-600 mb-2">Our team is ready to help you</p>
                                    <p className="font-bold text-lg text-red-600">{setting.phone}</p>
                                    <p className="text-xs text-gray-500 mt-1">Monday-Friday, 9AM-5PM EST</p>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    {/* Professional Process Section */}
                    <div className="max-w-4xl mx-auto mt-12 bg-white rounded-xl shadow-md p-8 border border-gray-200">
                        <div className="text-center mb-8">
                            <h3 className="text-2xl font-bold text-gray-800 mb-2">Our Quote Process</h3>
                            <p className="text-gray-600">Simple, transparent, and efficient</p>
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <div className="text-center p-4">
                                <div className="flex items-center justify-center w-16 h-16 mx-auto rounded-full bg-red-100 text-red-600 mb-4">
                                    <span className="text-xl font-bold">1</span>
                                </div>
                                <h4 className="font-semibold text-gray-800 mb-2">Submit Request</h4>
                                <p className="text-gray-600 text-sm">Provide details about your project requirements</p>
                            </div>
                            
                            <div className="text-center p-4">
                                <div className="flex items-center justify-center w-16 h-16 mx-auto rounded-full bg-red-100 text-red-600 mb-4">
                                    <span className="text-xl font-bold">2</span>
                                </div>
                                <h4 className="font-semibold text-gray-800 mb-2">Review & Analysis</h4>
                                <p className="text-gray-600 text-sm">Our team evaluates your requirements and prepares a proposal</p>
                            </div>
                            
                            <div className="text-center p-4">
                                <div className="flex items-center justify-center w-16 h-16 mx-auto rounded-full bg-red-100 text-red-600 mb-4">
                                    <span className="text-xl font-bold">3</span>
                                </div>
                                <h4 className="font-semibold text-gray-800 mb-2">Receive Proposal</h4>
                                <p className="text-gray-600 text-sm">Get a detailed quote with timeline and cost breakdown</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <Footer setting={setting} />
        </div>
    );
}