import React, { useEffect, useState } from 'react';
import { Head } from '@inertiajs/react';
import Navbar from '@/Components/Navbar';
import Footer from '@/Components/Footer';
import { motion } from 'framer-motion';
import Preloader from '@/Components/Preloader';

const containerVariants = {
    hidden: { opacity: 0 },
    visible: {
        opacity: 1,
        transition: {
            staggerChildren: 0.2,
            delayChildren: 0.3
        }
    }
};

const itemVariants = {
    hidden: { y: 30, opacity: 0, scale: 0.9 },
    visible: {
        y: 0,
        opacity: 1,
        scale: 1,
        transition: { duration: 0.6, ease: "easeOut" }
    }
};

const TeamMemberCard = ({ member, addToCart, isAdded }) => {
    return (
        <motion.div
            className="relative group"
            variants={itemVariants}
            whileHover={{
                y: -10,
                transition: {
                    type: "spring",
                    stiffness: 300,
                    damping: 15
                }
            }}
        >
            <div className="absolute -inset-2 bg-gradient-to-r from-yellow-400 to-red-500 rounded-2xl opacity-20 group-hover:opacity-30 transition-all duration-400 blur-sm"></div>

            <div className="relative bg-white rounded-2xl shadow-lg overflow-hidden border border-red-100 transition-all duration-300 group-hover:shadow-xl">
                <div className="relative">
                    <div className="absolute top-0 left-0 w-12 h-12 border-t-3 border-l-3 border-red-500 rounded-tl-xl opacity-60 z-10"></div>
                    <div className="absolute top-0 right-0 w-12 h-12 border-t-3 border-r-3 border-yellow-500 rounded-tr-xl opacity-60 z-10"></div>

                    <div className="relative w-full h-64 overflow-hidden">
                        <motion.div
                            className="absolute inset-0 border-4 border-white shadow-md"
                            whileHover={{
                                scale: 1.02,
                                transition: {
                                    type: "spring",
                                    stiffness: 400,
                                    damping: 10
                                }
                            }}
                        ></motion.div>

                        <motion.img
                            src={member.image_url}
                            alt={member.name}
                            className="object-cover"
                            whileHover={{
                                scale: 1.05,
                                transition: {
                                    type: "spring",
                                    stiffness: 400,
                                    damping: 10
                                }
                            }}
                        />

                        <div className="absolute inset-0 bg-gradient-to-t from-black/20 to-transparent"></div>

                        <motion.div
                            className="absolute top-4 right-4 w-10 h-10 rounded-full bg-red-100 flex items-center justify-center text-red-500 shadow-md z-10"
                            whileHover={{
                                rotate: 15,
                                scale: 1.2,
                                transition: {
                                    type: "spring",
                                    stiffness: 500,
                                    damping: 10
                                }
                            }}
                        >
                            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
                                <path fillRule="evenodd" d="M3.172 5.172a4 4 0 015.656 0L10 6.343l1.172-1.171a4 4 0 115.656 5.656L10 17.657l-6.828-6.829a4 4 0 010-5.656z" clipRule="evenodd" />
                            </svg>
                        </motion.div>
                    </div>

                    <div className="absolute bottom-0 left-0 w-full h-1 bg-gradient-to-r from-red-600 to-yellow-500"></div>
                </div>

                <div className="p-6">
                    <h3 className="text-xl font-bold text-gray-800 mb-1">{member.name}</h3>

                    <motion.div
                        className="inline-block px-4 py-2 mb-4 text-sm font-semibold rounded-full bg-gradient-to-r from-red-600 to-yellow-500 text-white shadow-sm"
                        whileHover={{
                            scale: 1.1,
                            transition: {
                                type: "spring",
                                stiffness: 500,
                                damping: 10
                            }
                        }}
                    >
                        {member.position}
                    </motion.div>
                    <button
                        onClick={() => addToCart(member)}
                        disabled={isAdded}
                        className="w-full mt-4 px-4 py-2 rounded-lg text-white bg-gradient-to-r from-red-600 to-yellow-500 hover:from-red-700 hover:to-yellow-600 disabled:opacity-50"
                    >
                        {isAdded ? 'Added' : 'Add to Cart'}
                    </button>
                </div>

                <div className="absolute bottom-0 left-0 w-full h-1 bg-gradient-to-r from-red-600 to-yellow-500"></div>
            </div>

            <div className="absolute -bottom-4 left-1/2 transform -translate-x-1/2 w-12 h-1 bg-gradient-to-r from-red-600 to-yellow-500 rounded-full"></div>
            <div className="absolute -bottom-6 left-1/2 transform -translate-x-1/2 w-8 h-1 bg-gradient-to-r from-red-600 to-yellow-500 rounded-full"></div>
        </motion.div>
    );
};

const Team = ({ teams, setting }) => {
    const [loading, setLoading] = useState(true);
    const [addedItems, setAddedItems] = useState([]);
    const { post } = useForm();

    const addToCart = (member) => {
        post(route('cart.add'), {
            item_id: member.id,
            item_type: 'team',
            quantity: 1,
            onSuccess: () => {
                setAddedItems([...addedItems, member.id]);
            }
        });
    };
            
    useEffect(() => {
        const timer = setTimeout(() => {
            setLoading(false);
        }, 1000);

        return () => clearTimeout(timer);
    }, []);

    if (loading) {
        return <Preloader />;
    }
    return (
        <div className="min-h-screen bg-white mt-24 font-sans antialiased">
            <Head title="Our Team" />
            <Navbar setting={setting} />

            {/* Professional Hero Section - Matching the provided structure */}
            <div className="relative bg-gradient-to-r from-red-700 to-red-800 py-16 text-white">
                <div className="container mx-auto px-6">
                    <div className="max-w-3xl">
                        <h1 className="text-4xl md:text-5xl font-bold mb-4">Our Team</h1>
                        <p className="text-xl text-red-100 max-w-2xl">
                            Meet the talented individuals behind our success. Our diverse team brings together expertise, creativity, and passion to deliver exceptional results.
                        </p>
                    </div>
                </div>
                
                {/* Subtle decorative elements */}
                <div className="absolute top-0 right-0 w-96 h-96 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
            </div>

            <section className="py-24 bg-gradient-to-b from-white to-red-50 relative overflow-hidden" id="team">
                <div className="absolute inset-0 bg-[url('data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHdpZHRoPSIxMDAlIiBoZWlnaHQ9IjEwMCUiPjxkZWZzPjxwYXR0ZXJuIGlkPSJwYXR0ZXJuIiB3aWR0aD0iNDAiIGhlaWdodD0iNDAiIHBhdHRlcm5Vbml0cz0idXNlclNwYWNlT25Vc2UiIHBhdHRlcm5UcmFuc2Zvcm09InJvdGF0ZSg0NSkiPjxyZWN0IHg9IjAiIHk9IjAiIHdpZHRoPSIyMCIgaGVpZ2h0PSIyMCIgZmlsbD0icmdiYSgyMDQsMCwwLDAuMDMpIi8+PC9wYXR0ZXJuPjwvZGVmcz48cmVjdCB3aWR0aD0iMTAwJSIgaGVpZ2h0PSIxMDAlIiBmaWxsPSJ1cmwoI3BhdHRlcm4pIi8+PC9zdmc+')] opacity-10"></div>

                <div className="container mx-auto px-6 relative z-10">
                    {/* Section Header */}
                    <motion.div
                        className="text-center mb-16"
                        initial={{ opacity: 0, y: 20 }}
                        whileInView={{ opacity: 1, y: 0 }}
                        transition={{ duration: 0.7 }}
                        viewport={{ once: true }}
                    >
                        <h2 className="text-3xl font-bold text-gray-800 mb-4">Leadership & Expertise</h2>
                        <p className="text-lg text-gray-600 max-w-2xl mx-auto">
                            Our team consists of industry experts with diverse backgrounds and a shared commitment to excellence.
                        </p>
                    </motion.div>

                    <motion.div
                        className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8"
                        variants={containerVariants}
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true, amount: 0.2 }}
                    >
                        {teams && teams.length > 0 ? (
                            teams.map((member) => (
                                <TeamMemberCard key={member.id} member={member} addToCart={addToCart} isAdded={addedItems.includes(member.id)} />
                            ))
                        ) : (
                            <div className="col-span-full text-center py-10">
                                <p className="text-xl text-gray-600">No team members available at the moment.</p>
                            </div>
                        )}
                    </motion.div>

                    {/* Team Values Section */}
                    <motion.div
                        className="mt-24 bg-gradient-to-r from-red-700 to-red-800 rounded-2xl p-10 md:p-16 text-white shadow-xl overflow-hidden"
                        initial={{ opacity: 0, scale: 0.95 }}
                        whileInView={{ opacity: 1, scale: 1 }}
                        transition={{ duration: 0.7 }}
                        viewport={{ once: true }}
                    >
                        {/* Decorative elements */}
                        <div className="absolute top-0 right-0 w-64 h-64 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
                        <div className="absolute bottom-0 left-0 w-64 h-64 bg-yellow-400 rounded-full mix-blend-soft-light filter blur-3xl opacity-20"></div>
                        
                        <div className="relative z-10">
                            <h3 className="text-3xl font-bold text-center mb-6">Our Team Values</h3>
                            
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-8">
                                <div className="text-center">
                                    <div className="flex items-center justify-center w-16 h-16 mx-auto rounded-full bg-red-900 text-yellow-400 mb-4">
                                        <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                                        </svg>
                                    </div>
                                    <h4 className="text-xl font-bold mb-2">Innovation</h4>
                                    <p className="text-red-100">We embrace creativity and new ideas to solve complex challenges.</p>
                                </div>
                                
                                <div className="text-center">
                                    <div className="flex items-center justify-center w-16 h-16 mx-auto rounded-full bg-red-900 text-yellow-400 mb-4">
                                        <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M17 20h5v-2a3 3 0 00-5.356-1.857M17 20H7m10 0v-2c0-.656-.126-1.283-.356-1.857M7 20H2v-2a3 3 0 015.356-1.857M7 20v-2c0-.656.126-1.283.356-1.857m0 0a5.002 5.002 0 019.288 0M15 7a3 3 0 11-6 0 3 3 0 016 0zm6 3a2 2 0 11-4 0 2 2 0 014 0zM7 10a2 2 0 11-4 0 2 2 0 014 0z" />
                                        </svg>
                                    </div>
                                    <h4 className="text-xl font-bold mb-2">Collaboration</h4>
                                    <p className="text-red-100">We work together to achieve the best possible outcomes for our clients.</p>
                                </div>
                                
                                <div className="text-center">
                                    <div className="flex items-center justify-center w-16 h-16 mx-auto rounded-full bg-red-900 text-yellow-400 mb-4">
                                        <svg className="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z" />
                                        </svg>
                                    </div>
                                    <h4 className="text-xl font-bold mb-2">Integrity</h4>
                                    <p className="text-red-100">We operate with honesty and transparency in all that we do.</p>
                                </div>
                            </div>
                        </div>
                    </motion.div>
                </div>
            </section>

            <Footer setting={setting} />
        </div>
    );
};

export default Team;