import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link, router } from '@inertiajs/react';
import { useState, useEffect } from 'react';

// Helper function to format dates
const formatDate = (dateString) => {
    if (!dateString) return 'N/A';
    
    const date = new Date(dateString);
    const options = { 
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
    };
    return date.toLocaleDateString('en-US', options);
};

// Helper function to format date with time
const formatDateTime = (dateString) => {
    if (!dateString) return 'N/A';
    
    const date = new Date(dateString);
    const options = { 
        year: 'numeric', 
        month: 'short', 
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit'
    };
    return date.toLocaleDateString('en-US', options);
};

// Helper function to get relative time
const getRelativeTime = (dateString) => {
    if (!dateString) return 'N/A';
    
    const date = new Date(dateString);
    const now = new Date();
    const diffInSeconds = Math.floor((now - date) / 1000);
    
    if (diffInSeconds < 60) return 'Just now';
    if (diffInSeconds < 3600) return `${Math.floor(diffInSeconds / 60)} minutes ago`;
    if (diffInSeconds < 86400) return `${Math.floor(diffInSeconds / 3600)} hours ago`;
    if (diffInSeconds < 604800) return `${Math.floor(diffInSeconds / 86400)} days ago`;
    if (diffInSeconds < 2592000) return `${Math.floor(diffInSeconds / 604800)} weeks ago`;
    if (diffInSeconds < 31536000) return `${Math.floor(diffInSeconds / 2592000)} months ago`;
    return `${Math.floor(diffInSeconds / 31536000)} years ago`;
};

// Helper function to format appointment date
const formatAppointmentDate = (dateString, timeString) => {
    if (!dateString) return 'N/A';
    
    const date = new Date(dateString);
    const options = { 
        weekday: 'long',
        year: 'numeric', 
        month: 'long', 
        day: 'numeric' 
    };
    const formattedDate = date.toLocaleDateString('en-US', options);
    
    if (timeString) {
        return `${formattedDate} at ${timeString}`;
    }
    
    return formattedDate;
};

export default function Dashboard({ auth, recentOrders, recentBookings, unreadNotifications, completedBookingsCount, pendingBookingsCount }) {
    const [activeTab, setActiveTab] = useState('overview');

    // Convert unreadNotifications to a format compatible with existing notification structure
    const formattedNotifications = unreadNotifications.map(notification => ({
        id: notification.id,
        message: notification.data.message || 'New notification',
        time: notification.created_at,
        read: notification.read_at !== null,
    }));

    const unreadCount = formattedNotifications.filter(n => !n.read).length;

     const markAsRead = (id) => {
        router.post(route('admin.notifications.markAsRead', id), {}, {
            onSuccess: () => {
                // Inertia will re-render the component with updated props
            },
            // Optionally, handle error here
        });
    };

    // Sample appointment data with proper dates
    const nextAppointment = recentBookings.length > 0 ? {
        doctor: recentBookings[0].service_name || recentBookings[0].school_service_name || recentBookings[0].event_name,
        date: recentBookings[0].booking_date,
        time: recentBookings[0].booking_time,
        type: recentBookings[0].service_name ? 'Service Booking' : (recentBookings[0].school_service_name ? 'School Service Booking' : 'Event Booking'),
        room: recentBookings[0].location || 'N/A'
    } : null;

    // Use recentBookings for upcomingAppointments
    const upcomingAppointments = recentBookings.map(booking => ({
        id: booking.id,
        type: booking.service_name ? 'Service Booking' : (booking.school_service_name ? 'School Service Booking' : 'Event Booking'),
        doctor: booking.service_name || booking.school_service_name || booking.event_name,
        date: booking.booking_date,
        time: booking.booking_time,
        location: booking.location || 'N/A',
        status: booking.status,
    }));

    const recentDocuments = [
        {
            id: 1,
            name: 'Lab Results - Blood Work',
            date: '2025-11-01',
            type: 'Lab Results'
        },
        {
            id: 2,
            name: 'X-Ray Report',
            date: '2025-10-15',
            type: 'Imaging'
        },
        {
            id: 3,
            name: 'Prescription Refill',
            date: '2025-10-05',
            type: 'Prescription'
        }
    ];

    return (
        <AdminLayout>
            <Head title="User Dashboard" />

            <div className="py-4 sm:py-6 bg-gray-50 min-h-screen">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Welcome Section */}
                    <div className="mb-6 sm:mb-8 bg-gradient-to-r from-green-500 to-emerald-600 rounded-xl shadow-xl p-4 sm:p-6 lg:p-8 text-white relative overflow-hidden">
                        <div className="absolute top-0 right-0 -mt-4 -mr-4 h-32 w-32 rounded-full bg-white opacity-10"></div>
                        <div className="absolute bottom-0 left-0 -mb-4 -ml-4 h-40 w-40 rounded-full bg-white opacity-10"></div>
                        
                        <div className="flex flex-col md:flex-row md:items-center md:justify-between relative z-10">
                            <div>
                                <h1 className="text-2xl sm:text-3xl font-bold mb-2">Welcome back, {auth.user.name}!</h1>
                                <p className="text-green-100 text-sm sm:text-base">Here's what's happening with your account today.</p>
                            </div>
                            <div className="mt-4 md:mt-0">
                                <div className="flex items-center bg-white bg-opacity-20 rounded-xl p-3 sm:p-4 backdrop-blur-sm">
                                    <div className="flex-shrink-0 h-12 w-12 sm:h-16 sm:w-16 bg-white bg-opacity-30 rounded-full flex items-center justify-center">
                                        <span className="text-lg sm:text-2xl font-bold">{auth.user.name.charAt(0)}</span>
                                    </div>
                                    <div className="ml-3 sm:ml-4">
                                        <p className="text-xs sm:text-sm font-medium text-green-100">Member since</p>
                                        <p className="text-sm sm:text-lg font-semibold">{formatDate(auth.user.created_at)}</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Stats Cards */}
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 sm:gap-6 mb-6 sm:mb-8">
                        <div className="bg-white overflow-hidden shadow-lg rounded-xl transition-all duration-300 hover:shadow-xl hover:-translate-y-1">
                            <div className="p-4 sm:p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0 bg-green-500 rounded-lg p-2 sm:p-3">
                                        <svg className="h-5 w-5 sm:h-6 sm:w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7V3m8 4V3m-9 8h10M5 21h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v12a2 2 0 002 2z" />
                                        </svg>
                                    </div>
                                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">Upcoming Appointments</dt>
                                            <dd className="text-lg sm:text-lg font-medium text-gray-900">{recentBookings.length}</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                            <div className="bg-gray-50 px-4 sm:px-5 py-2 sm:py-3">
                                <div className="text-sm">
                                    <Link href="/appointments" className="font-medium text-green-600 hover:text-green-800 transition-colors">
                                        View all
                                    </Link>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-lg rounded-xl transition-all duration-300 hover:shadow-xl hover:-translate-y-1">
                            <div className="p-4 sm:p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0 bg-emerald-500 rounded-lg p-2 sm:p-3">
                                        <svg className="h-5 w-5 sm:h-6 sm:w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                        </svg>
                                    </div>
                                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">Completed</dt>
                                            <dd className="text-lg sm:text-lg font-medium text-gray-900">{completedBookingsCount}</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                            <div className="bg-gray-50 px-4 sm:px-5 py-2 sm:py-3">
                                <div className="text-sm">
                                    <span className="font-medium text-emerald-600">+2 from last month</span>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-lg rounded-xl transition-all duration-300 hover:shadow-xl hover:-translate-y-1">
                            <div className="p-4 sm:p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0 bg-yellow-500 rounded-lg p-2 sm:p-3">
                                        <svg className="h-5 w-5 sm:h-6 sm:w-6 text-white" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                                        </svg>
                                    </div>
                                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">Pending</dt>
                                            <dd className="text-lg sm:text-lg font-medium text-gray-900">{pendingBookingsCount}</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                            <div className="bg-gray-50 px-4 sm:px-5 py-2 sm:py-3">
                                <div className="text-sm">
                                    <span className="font-medium text-yellow-600">Action required</span>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Tabs Section */}
                    <div className="bg-white shadow-xl rounded-xl mb-6 sm:mb-8 overflow-hidden">
                        <div className="border-b border-gray-200">
                            <nav className="-mb-px flex overflow-x-auto">
                                <button
                                    onClick={() => setActiveTab('overview')}
                                    className={`py-3 sm:py-4 px-3 sm:px-6 border-b-2 font-medium text-xs sm:text-sm transition-colors whitespace-nowrap ${
                                        activeTab === 'overview'
                                            ? 'border-green-500 text-green-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    Overview
                                </button>
                                <button
                                    onClick={() => setActiveTab('appointments')}
                                    className={`py-3 sm:py-4 px-3 sm:px-6 border-b-2 font-medium text-xs sm:text-sm transition-colors whitespace-nowrap ${
                                        activeTab === 'appointments'
                                            ? 'border-green-500 text-green-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    Appointments
                                </button>
                                <button
                                    onClick={() => setActiveTab('notifications')}
                                    className={`py-3 sm:py-4 px-6 sm:px-6 border-b-2 font-medium text-xs sm:text-sm relative transition-colors whitespace-nowrap ${
                                        activeTab === 'notifications'
                                            ? 'border-green-500 text-green-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    Notifications &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;
                                    {unreadCount > 0 && (
                                        <span className="absolute top-3 sm:top-4 right-6 sm:right-4 inline-flex items-center justify-center px-1.5 sm:px-2 py-0.5 text-xs font-bold leading-none text-white transform translate-x-1/2 -translate-y-1/2 bg-red-500 rounded-full animate-pulse">
                                            {unreadCount}
                                        </span>
                                    )}
                                </button>
                            </nav>
                        </div>

                        <div className="p-4 sm:p-6">
                            {activeTab === 'overview' && (
                                <div>
                                    <h3 className="text-lg font-medium text-gray-900 mb-4">Account Overview</h3>
                                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                                        <div className="bg-gradient-to-br from-green-50 to-emerald-50 p-3 sm:p-4 rounded-xl border border-green-100 transition-all duration-300 hover:shadow-md">
                                            <h4 className="text-xs sm:text-sm font-medium text-gray-500 mb-2">Next Appointment</h4>
                                            {nextAppointment ? (
                                                <>
                                                    <p className="text-base sm:text-lg font-semibold text-gray-900 truncate">{nextAppointment.doctor}</p>
                                                    <p className="text-xs sm:text-sm text-gray-600">{formatAppointmentDate(nextAppointment.date, nextAppointment.time)}</p>
                                                    <p className="text-xs sm:text-sm text-gray-600">{nextAppointment.type}</p>
                                                </>
                                            ) : (
                                                <p className="text-sm text-gray-600">No upcoming appointments.</p>
                                            )}
                                        </div>
                                        <div className="bg-gradient-to-br from-green-50 to-emerald-50 p-3 sm:p-4 rounded-xl border border-green-100 transition-all duration-300 hover:shadow-md">
                                            <h4 className="text-xs sm:text-sm font-medium text-gray-500 mb-2">Recent Orders</h4>
                                            {recentOrders.length > 0 ? (
                                                <ul className="space-y-1 sm:space-y-2">
                                                    {recentOrders.map(order => (
                                                        <li key={order.id} className="flex justify-between items-center">
                                                            <p className="text-xs sm:text-sm font-semibold text-gray-900 truncate">Order #{order.id}</p>
                                                            <p className="text-xs sm:text-sm text-green-600">₵{order.total_amount}</p>
                                                        </li>
                                                    ))}
                                                </ul>
                                            ) : (
                                                <p className="text-xs sm:text-sm text-gray-600">No recent orders.</p>
                                            )}
                                            <Link href={route('user.orders.index')} className="text-xs sm:text-sm text-green-600 hover:text-green-800 transition-colors mt-2 block">View All Orders</Link>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {activeTab === 'appointments' && (
                                <div>
                                    <h3 className="text-lg font-medium text-gray-900 mb-4">Upcoming Appointments</h3>
                                    <div className="space-y-3 sm:space-y-4">
                                        {recentBookings.length === 0 ? (
                                            <p className="text-gray-600">No upcoming appointments.</p>
                                        ) : (
                                            recentBookings.map((appointment) => (
                                                <div key={appointment.id} className={`border-l-4 ${
                                                    appointment.status === 'confirmed' ? 'border-green-500 bg-gradient-to-r from-green-50 to-transparent' : 
                                                    appointment.status === 'video' ? 'border-emerald-500 bg-gradient-to-r from-emerald-50 to-transparent' : 
                                                    'border-yellow-500 bg-gradient-to-r from-yellow-50 to-transparent'
                                                } p-3 sm:p-4 rounded-r-lg transition-all duration-300 hover:shadow-md`}>
                                                    <div className="flex flex-col sm:flex-row sm:justify-between">
                                                        <div className="mb-2 sm:mb-0">
                                                            <h4 className="text-sm sm:text-md font-medium text-gray-900 truncate">{appointment.service_name || appointment.school_service_name || appointment.event_name}</h4>
                                                            <p className="text-xs sm:text-sm text-gray-600">{formatAppointmentDate(appointment.booking_date, appointment.booking_time)}</p>
                                                            <p className="text-xs sm:text-sm text-gray-600">Status: {appointment.status}</p>
                                                        </div>
                                                        <div className="flex flex-row sm:flex-col space-x-2 sm:space-x-0 sm:space-y-2">
                                    {/* Action buttons based on status */}
                                    {appointment.status === 'pending' && (
                                        <button
                                            onClick={() => router.post(route('user.bookings.confirm', appointment.id))}
                                            className="px-3 py-1 bg-yellow-600 text-white text-xs sm:text-sm rounded-lg hover:bg-yellow-700 transition-colors"
                                        >
                                            Confirm
                                        </button>
                                    )}
                                    {appointment.status === 'confirmed' && (
                                        <button
                                            onClick={() => router.post(route('user.bookings.reschedule', appointment.id))}
                                            className="px-3 py-1 bg-green-600 text-white text-xs sm:text-sm rounded-lg hover:bg-green-700 transition-colors"
                                        >
                                            Reschedule
                                        </button>
                                    )}
                                    <button
                                        onClick={() => {
                                            if (window.confirm('Are you sure you want to cancel this booking?')) {
                                                router.post(route('user.bookings.cancel', appointment.id));
                                            }
                                        }}
                                        className="px-3 py-1 bg-white text-gray-600 border border-gray-300 text-xs sm:text-sm rounded-lg hover:bg-gray-50 transition-colors"
                                    >
                                        Cancel
                                    </button>
                                </div>
                                                    </div>
                                                </div>
                                            ))
                                        )}
                                    </div>
                                    <div className="mt-6 text-center">
                                        <Link href={route('user.bookings.index')} className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 transition-colors">
                                            View All Appointments
                                        </Link>
                                    </div>
                                </div>
                            )}

                            {activeTab === 'notifications' && (
                                <div>
                                    <h3 className="text-lg font-medium text-gray-900 mb-4">Notifications</h3>
                                    <div className="space-y-3 sm:space-y-4">
                                        {formattedNotifications.length === 0 ? (
                                            <p className="text-gray-600">No new notifications.</p>
                                        ) : (
                                            formattedNotifications.map((notification) => (
                                                <div key={notification.id} className={`p-3 sm:p-4 rounded-lg transition-all duration-300 ${notification.read ? 'bg-gray-50' : 'bg-gradient-to-r from-green-50 to-emerald-50 border border-green-100'}`}>
                                                    <div className="flex flex-col sm:flex-row sm:justify-between">
                                                        <div className="flex mb-2 sm:mb-0">
                                                            <div className="flex-shrink-0">
                                                                <svg className={`h-5 w-5 sm:h-6 sm:w-6 ${notification.read ? 'text-gray-400' : 'text-green-500'}`} fill="none" viewBox="0 0 24 24" stroke="currentColor">
                                                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 17h5l-1.405-1.405A2.032 2.032 0 0118 14.158V11a6.002 6.002 0 00-4-5.659V5a2 2 0 10-4 0v.341C7.67 6.165 6 8.388 6 11v3.159c0 .538-.214 1.055-.595 1.436L4 17h5m6 0v1a3 3 0 11-6 0v-1m6 0H9" />
                                                                </svg>
                                                            </div>
                                                            <div className="ml-3">
                                                                <p className={`text-xs sm:text-sm ${notification.read ? 'text-gray-700' : 'text-gray-900 font-medium'}`}>
                                                                    {notification.message}
                                                                </p>
                                                                <p className="text-xs text-gray-500 mt-1">{getRelativeTime(notification.time)}</p>
                                                            </div>
                                                        </div>
                                                        {!notification.read && (
                                                            <button
                                                                onClick={() => markAsRead(notification.id)}
                                                                className="text-xs text-green-600 hover:text-green-800 transition-colors mt-2 sm:mt-0"
                                                            >
                                                                Mark as read
                                                            </button>
                                                        )}
                                                    </div>
                                                </div>
                                            ))
                                        )}
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}