import React, { useState } from 'react';
import { Head, Link, useForm } from '@inertiajs/react';
import { 
    ClipboardDocumentCheckIcon,
    UserIcon,
    CalendarIcon,
    ClockIcon,
    CurrencyDollarIcon,
    CheckCircleIcon,
    ExclamationCircleIcon,
    XCircleIcon,
    CreditCardIcon,
    ChatBubbleLeftIcon,
    PhoneIcon,
    EnvelopeIcon,
    FunnelIcon,
    MagnifyingGlassIcon,
    ArrowDownTrayIcon,
    EyeIcon
} from '@heroicons/react/24/outline';
import AdminLayout from '@/Layouts/AdminLayout';

const BookingCard = ({ bookingDetail }) => {
    if (!bookingDetail.booking) {
        console.warn('BookingCard received bookingDetail with null booking:', bookingDetail);
        return null; // Don't render if booking is null
    }

    console.log(bookingDetail?.booking?.user)
    // Extract nested objects from the booking detail
    const booking = bookingDetail.booking || {};
    const service = bookingDetail.service;
    const schoolService = bookingDetail.school_service; // Assuming school_service relationship
    const user = bookingDetail.booking?.user || {};
    const [showDetails, setShowDetails] = useState(false);
    
    const { post, processing } = useForm();

    const updateStatus = (status) => {
        post(route('user.doctors.assigned-bookings.update-approval-status', { booking: booking.id, status }), {
            preserveScroll: true,
        });
    };

    // Parse the payload if it exists to get payment details
    let paymentDetails = null;
    if (booking && booking.payload) {
        try {
            paymentDetails = JSON.parse(booking.payload);
        } catch (e) {
            console.error('Error parsing payment payload:', e);
        }
    }

    // Function to get status icon and color
    const getStatusInfo = (status) => {
        switch (status) {
            case 'approved':
                return { 
                    icon: <CheckCircleIcon className="h-5 w-5" />, 
                    color: 'bg-green-100 text-green-800 border-green-200',
                    bg: 'bg-green-50'
                };
            case 'pending':
                return { 
                    icon: <ExclamationCircleIcon className="h-5 w-5" />, 
                    color: 'bg-yellow-100 text-yellow-800 border-yellow-200',
                    bg: 'bg-yellow-50'
                };
                        case 'declined':
                            return {
                                icon: <XCircleIcon className="h-5 w-5" />,
                                color: 'bg-red-100 text-red-800 border-red-200',
                                bg: 'bg-red-50'
                            };
                        case 'completed':
                            return {
                                icon: <ClipboardDocumentCheckIcon className="h-5 w-5" />,
                                color: 'bg-teal-100 text-teal-800 border-teal-200',
                                bg: 'bg-teal-50'
                            };            default:
                return { 
                    icon: <ExclamationCircleIcon className="h-5 w-5" />, 
                    color: 'bg-gray-100 text-gray-800 border-gray-200',
                    bg: 'bg-gray-50'
                };
        }
    };

    const statusInfo = bookingDetail ? getStatusInfo(bookingDetail.doctor_approval_status) : getStatusInfo('unknown');

    // Format date and time with null checks
    const formatDate = (dateString) => {
        if (!dateString) return 'Not specified';
        const options = { year: 'numeric', month: 'long', day: 'numeric' };
        return new Date(dateString).toLocaleDateString(undefined, options);
    };

    const formatTime = (timeString) => {
        if (!timeString) return 'Not specified';
        
        try {
            const [hours, minutes] = timeString.split(':');
            const hour = parseInt(hours, 10);
            const ampm = hour >= 12 ? 'PM' : 'AM';
            const formattedHour = hour % 12 || 12;
            return `${formattedHour}:${minutes} ${ampm}`;
        } catch (e) {
            console.error('Error formatting time:', e);
            return 'Invalid time format';
        }
    };

    // Extract phone number from user or payment details
    const phoneNumber = user?.mobile_number || 
                       (paymentDetails?.authorization?.mobile_money_number) || 
                       'Not provided';

    const bookingTitle = service ? service.title : (schoolService ? schoolService.title : 'Booking not available');
    const bookingType = service ? service.type : (schoolService ? schoolService.type : 'N/A');

    return (
        <div className={`bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden transition-all duration-300 hover:shadow-lg ${statusInfo.bg}`}>
            <div className="p-6">
                <div className="flex items-start justify-between">
                    <div className="flex-1 min-w-0">
                        <div className="flex items-center justify-between mb-4">
                            <div className="flex items-center">
                                <h3 className="text-xl font-semibold text-gray-900 truncate">
                                    {bookingTitle}
                                </h3>
                                <span className={`ml-3 inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ${
                                    bookingType === 'paid' ? 'bg-blue-100 text-blue-800' : 'bg-green-100 text-green-800'
                                }`}>
                                    {bookingType || 'free'}
                                </span>
                            </div>
                            <button
                                onClick={() => setShowDetails(!showDetails)}
                                className="text-gray-400 hover:text-gray-600 transition-colors"
                            >
                                <EyeIcon className="h-5 w-5" />
                            </button>
                        </div>
                        
                        <div className="flex items-center mb-4">
                            <div className="flex-shrink-0 mr-3">
                                {user ? (
                                    <div className="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center">
                                        <span className="text-sm font-medium text-gray-700">
                                            {bookingDetail?.booking?.user ? bookingDetail?.booking?.user.name.charAt(0).toUpperCase() : 'U'}
                                        </span>
                                    </div>
                                ) : (
                                    <div className="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center">
                                        <UserIcon className="h-6 w-6 text-gray-400" />
                                    </div>
                                )}
                            </div>
                            <div>
                                <p className="text-sm font-medium text-gray-900">
                                    {bookingDetail?.booking?.user ? bookingDetail?.booking?.user.name : 'User information not available'}
                                </p>
                                <div className="flex items-center mt-1 text-xs text-gray-500">
                                    {user?.email && (
                                        <span className="flex items-center mr-3">
                                            <EnvelopeIcon className="h-3 w-3 mr-1" />
                                            {user.email}
                                        </span>
                                    )}
                                    <span className="flex items-center">
                                        <PhoneIcon className="h-3 w-3 mr-1" />
                                        {phoneNumber}
                                    </span>
                                </div>
                            </div>
                        </div>
                        
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                            <div className="flex items-center text-sm text-gray-600">
                                <div className="flex-shrink-0 mr-2 p-2 bg-gray-100 rounded-lg">
                                    <CalendarIcon className="h-4 w-4 text-gray-600" />
                                </div>
                                <div>
                                    <p className="font-medium">Date</p>
                                    <p>{booking ? formatDate(booking.booking_date) : 'Not specified'}</p>
                                </div>
                            </div>
                            <div className="flex items-center text-sm text-gray-600">
                                <div className="flex-shrink-0 mr-2 p-2 bg-gray-100 rounded-lg">
                                    <ClockIcon className="h-4 w-4 text-gray-600" />
                                </div>
                                <div>
                                    <p className="font-medium">Time</p>
                                    <p>{formatTime(bookingDetail.booking_time)}</p>
                                </div>
                            </div>
                            <div className="flex items-center text-sm text-gray-600">
                                <div className="flex-shrink-0 mr-2 p-2 bg-gray-100 rounded-lg">
                                    <CurrencyDollarIcon className="h-4 w-4 text-gray-600" />
                                </div>
                                <div>
                                    <p className="font-medium">Price</p>
                                    <p>{booking ? booking.price : (service ? service.price : (schoolService ? schoolService.price : 'N/A'))}</p>
                                </div>
                            </div>
                        </div>
                        
                        {showDetails && (
                            <div className="mt-4 pt-4 border-t border-gray-200 animate-in">
                                {booking && booking.notes && (
                                    <div className="mb-4">
                                        <h4 className="text-sm font-medium text-gray-900 mb-2">Notes</h4>
                                        <p className="text-sm text-gray-600 bg-gray-50 p-3 rounded-lg">{booking.notes}</p>
                                    </div>
                                )}
                                {service && service.description && (
                                    <div className="mb-4">
                                        <h4 className="text-sm font-medium text-gray-900 mb-2">Service Description</h4>
                                        <p className="text-sm text-gray-600 bg-gray-50 p-3 rounded-lg">{service.description}</p>
                                    </div>
                                )}
                                {schoolService && schoolService.description && (
                                    <div className="mb-4">
                                        <h4 className="text-sm font-medium text-gray-900 mb-2">School Service Description</h4>
                                        <p className="text-sm text-gray-600 bg-gray-50 p-3 rounded-lg">{schoolService.description}</p>
                                    </div>
                                )}
                                
                                {paymentDetails && (
                                    <div className="mb-4">
                                        <h4 className="text-sm font-medium text-gray-900 mb-2 flex items-center">
                                            <CreditCardIcon className="h-4 w-4 mr-1" />
                                            Payment Details
                                        </h4>
                                        <div className="bg-gray-50 p-3 rounded-lg">
                                            <div className="grid grid-cols-2 gap-2 text-sm">
                                                <div>
                                                    <span className="text-gray-500">Reference:</span>
                                                    <p className="font-medium">{paymentDetails.reference || 'N/A'}</p>
                                                </div>
                                                <div>
                                                    <span className="text-gray-500">Amount:</span>
                                                    <p className="font-medium">{paymentDetails.currency} {paymentDetails.amount / 100}</p>
                                                </div>
                                                <div>
                                                    <span className="text-gray-500">Method:</span>
                                                    <p className="font-medium">{paymentDetails.channel || 'N/A'}</p>
                                                </div>
                                                <div>
                                                    <span className="text-gray-500">Paid on:</span>
                                                    <p className="font-medium">{paymentDetails.paidAt ? new Date(paymentDetails.paidAt).toLocaleDateString() : 'N/A'}</p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                )}
                                
                                <div className="flex space-x-2">
                                    <button className="inline-flex items-center px-3 py-1.5 border border-gray-300 shadow-sm text-xs font-medium rounded text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500">
                                        <ChatBubbleLeftIcon className="h-4 w-4 mr-1" />
                                        Contact Patient
                                    </button>
                                    <button className="inline-flex items-center px-3 py-1.5 border border-gray-300 shadow-sm text-xs font-medium rounded text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500">
                                        <ArrowDownTrayIcon className="h-4 w-4 mr-1" />
                                        Download Details
                                    </button>
                                </div>
                            </div>
                        )}
                    </div>
                    
                    <div className="ml-4 flex-shrink-0">
                        <span className={`inline-flex items-center px-3 py-1.5 rounded-full text-sm font-medium border ${statusInfo.color}`}>
                            {statusInfo.icon}
                            <span className="ml-1.5">{bookingDetail ? bookingDetail.doctor_approval_status : 'Unknown'}</span>
                        </span>
                        <div className="flex space-x-2 mt-2">
                            {bookingDetail.doctor_approval_status === 'pending' && (
                                <>
                                    <button
                                        onClick={() => updateStatus('approved')}
                                        disabled={processing}
                                        className="inline-flex items-center px-3 py-1 border border-transparent text-sm leading-4 font-medium rounded-md shadow-sm text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 disabled:opacity-50"
                                    >
                                        Accept
                                    </button>
                                    <button
                                        onClick={() => updateStatus('declined')}
                                        disabled={processing}
                                        className="inline-flex items-center px-3 py-1 border border-transparent text-sm leading-4 font-medium rounded-md shadow-sm text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 disabled:opacity-50"
                                    >
                                        Decline
                                    </button>
                                </>
                            )}
                            {bookingDetail.doctor_approval_status === 'approved' && (
                                <button
                                    onClick={() => updateStatus('completed')}
                                    disabled={processing}
                                    className="inline-flex items-center px-3 py-1 border border-transparent text-sm leading-4 font-medium rounded-md shadow-sm text-white bg-teal-600 hover:bg-teal-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-teal-500 disabled:opacity-50"
                                >
                                    Mark as Completed
                                </button>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </div>
    );
};

export default function AssignedBookings({ auth, serviceBookings }) {
    const [searchTerm, setSearchTerm] = useState('');
    const [filterStatus, setFilterStatus] = useState('all');
    const [showFilters, setShowFilters] = useState(false);

    console.log(serviceBookings)
    
    // Filter bookings based on search term and status
    const filteredBookings = Object.values(serviceBookings.data || {}).filter(bookingDetail => {
        const booking = bookingDetail.booking;
        const service = bookingDetail.service;
        const user = bookingDetail.user;
        
        const matchesSearch = searchTerm === '' || 
            (service && service.title && service.title.toLowerCase().includes(searchTerm.toLowerCase())) ||
            (user && user.name && user.name.toLowerCase().includes(searchTerm.toLowerCase()));
        
        const matchesStatus = filterStatus === 'all' || (bookingDetail && bookingDetail.doctor_approval_status === filterStatus);
        
        return matchesSearch && matchesStatus;
    });

    return (
        <AdminLayout
            user={auth.user}
            header={
                <div className="flex justify-between items-center">
                    <h2 className="font-semibold text-xl text-gray-800 leading-tight">Assigned Bookings</h2>
                    <div className="flex items-center space-x-2">
                        <span className="text-sm text-gray-500">
                            {serviceBookings.total} total bookings
                        </span>
                    </div>
                </div>
            }
        >
            <Head title="Assigned Bookings" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Search and Filters */}
                    <div className="bg-white shadow-sm rounded-lg p-4 mb-6 border border-gray-200">
                        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                            <div className="relative flex-1 max-w-md">
                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <MagnifyingGlassIcon className="h-5 w-5 text-gray-400" />
                                </div>
                                <input
                                    type="text"
                                    className="block w-full pl-10 pr-3 py-2 border border-gray-300 rounded-md leading-5 bg-white placeholder-gray-500 focus:outline-none focus:placeholder-gray-400 focus:ring-1 focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                                    placeholder="Search bookings..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </div>
                            <div className="flex items-center space-x-2">
                                <button
                                    onClick={() => setShowFilters(!showFilters)}
                                    className="inline-flex items-center px-3 py-2 border border-gray-300 shadow-sm text-sm leading-4 font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                >
                                    <FunnelIcon className="h-4 w-4 mr-1" />
                                    Filters
                                </button>
                                <button className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md shadow-sm text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500">
                                    <ArrowDownTrayIcon className="h-4 w-4 mr-1" />
                                    Export
                                </button>
                            </div>
                        </div>
                        
                        {showFilters && (
                            <div className="mt-4 pt-4 border-t border-gray-200">
                                <div className="flex flex-wrap gap-2">
                                    <button
                                        onClick={() => setFilterStatus('all')}
                                        className={`px-3 py-1 rounded-full text-xs font-medium ${
                                            filterStatus === 'all' 
                                                ? 'bg-indigo-100 text-indigo-800' 
                                                : 'bg-gray-100 text-gray-800'
                                        }`}
                                    >
                                        All
                                    </button>
                                    <button
                                        onClick={() => setFilterStatus('approved')}
                                        className={`px-3 py-1 rounded-full text-xs font-medium ${
                                            filterStatus === 'approved' 
                                                ? 'bg-green-100 text-green-800' 
                                                : 'bg-gray-100 text-gray-800'
                                        }`}
                                    >
                                        Approved
                                    </button>
                                    <button
                                        onClick={() => setFilterStatus('pending')}
                                        className={`px-3 py-1 rounded-full text-xs font-medium ${
                                            filterStatus === 'pending' 
                                                ? 'bg-yellow-100 text-yellow-800' 
                                                : 'bg-gray-100 text-gray-800'
                                        }`}
                                    >
                                        Pending
                                    </button>
                                    <button
                                        onClick={() => setFilterStatus('declined')}
                                        className={`px-3 py-1 rounded-full text-xs font-medium ${
                                            filterStatus === 'declined' 
                                                ? 'bg-red-100 text-red-800' 
                                                : 'bg-gray-100 text-gray-800'
                                        }`}
                                    >
                                        Declined
                                    </button>
                                </div>
                            </div>
                        )}
                    </div>

                    {/* Bookings List */}
                    <div className="bg-white shadow-sm rounded-lg overflow-hidden border border-gray-200">
                        {filteredBookings.length > 0 ? (
                            <div className="divide-y divide-gray-200">
                                {filteredBookings.map((bookingDetail) => (
                                    <BookingCard key={bookingDetail.id} bookingDetail={bookingDetail} />
                                ))}
                            </div>
                        ) : (
                            <div className="text-center py-12">
                                <ClipboardDocumentCheckIcon className="mx-auto h-12 w-12 text-gray-400" />
                                <h3 className="mt-2 text-sm font-medium text-gray-900">No bookings found</h3>
                                <p className="mt-1 text-sm text-gray-500">
                                    {searchTerm || filterStatus !== 'all' 
                                        ? 'Try adjusting your search or filter criteria.' 
                                        : 'You have no service bookings assigned to you yet.'}
                                </p>
                            </div>
                        )}
                    </div>

                    {/* Pagination */}
                    {serviceBookings.links && serviceBookings.links.length > 3 && (
                        <div className="mt-6 flex justify-center">
                            <nav className="flex items-center space-x-1">
                                {serviceBookings.links.map((link, key) => (
                                    <Link
                                        key={key}
                                        href={link.url || '#'}
                                        className={`px-3 py-2 text-sm font-medium rounded-md transition-colors duration-150 ${
                                            link.active
                                                ? 'bg-indigo-600 text-white'
                                                : 'text-gray-500 bg-white hover:bg-gray-50 border border-gray-300'
                                        } ${!link.url ? 'opacity-50 cursor-not-allowed' : ''}`}
                                        dangerouslySetInnerHTML={{ __html: link.label }}
                                    />
                                ))}
                            </nav>
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}
