import AdminLayout from '@/Layouts/AdminLayout';
import { Head, Link } from '@inertiajs/react';
import { useState, useEffect } from 'react';
import { format } from 'date-fns';
import Pagination from '@/Components/Pagination';
import Modal from '@/Components/Modal';
import { 
    ShoppingBagIcon, 
    CheckCircleIcon, 
    ClockIcon, 
    XCircleIcon,
    MagnifyingGlassIcon,
    XMarkIcon,
    CurrencyDollarIcon,
    TruckIcon,
    DocumentTextIcon,
    ArrowDownTrayIcon,
    TrashIcon,
    UserIcon,
    CalendarIcon,
    CreditCardIcon,
    InformationCircleIcon,
    FunnelIcon,
    ChevronDownIcon
} from '@heroicons/react/24/outline';

export default function Index({ auth, orders }) {
    const [searchTerm, setSearchTerm] = useState('');
    const [filterStatus, setFilterStatus] = useState('all');
    const [filteredOrders, setFilteredOrders] = useState(orders?.data || []);
    const [showModal, setShowModal] = useState(false);
    const [selectedOrder, setSelectedOrder] = useState(null);
    const [activeTab, setActiveTab] = useState('all');
    const [isFilterExpanded, setIsFilterExpanded] = useState(false);
    const [isMobileView, setIsMobileView] = useState(false);

    // Detect screen size
    useEffect(() => {
        const handleResize = () => {
            setIsMobileView(window.innerWidth < 768);
        };
        
        handleResize(); // Initial check
        window.addEventListener('resize', handleResize);
        
        return () => window.removeEventListener('resize', handleResize);
    }, []);

    useEffect(() => {
        let filtered = orders?.data || [];

        // Filter by search term
        if (searchTerm) {
            filtered = filtered.filter(order =>
                (order.id && order.id.toLowerCase().includes(searchTerm.toLowerCase())) ||
                (order.status && order.status.toLowerCase().includes(searchTerm.toLowerCase()))
            );
        }

        // Filter by status
        if (filterStatus !== 'all') {
            filtered = filtered.filter(order => order.status === filterStatus);
        }

        // Filter by active tab
        if (activeTab !== 'all') {
            filtered = filtered.filter(order => order.status === activeTab);
        }

        setFilteredOrders(filtered);
    }, [searchTerm, filterStatus, activeTab, orders?.data]);

    const getStatusColor = (status) => {
        switch (status) {
            case 'completed':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'pending':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'cancelled':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'completed':
                return <CheckCircleIcon className="w-4 h-4" />;
            case 'pending':
                return <ClockIcon className="w-4 h-4" />;
            case 'cancelled':
                return <XCircleIcon className="w-4 h-4" />;
            default:
                return <InformationCircleIcon className="w-4 h-4" />;
        }
    };

    const viewDetails = (order) => {
        setSelectedOrder(order);
        setShowModal(true);
    };

    const closeModal = () => {
        setShowModal(false);
        setSelectedOrder(null);
    };

    const formatOrderId = (id) => {
        return id ? id.substring(0, 8) + '...' : 'N/A';
    };

    const formatAmount = (amount) => {
        return amount ? parseFloat(amount).toFixed(2) : '0.00';
    };

    // Calculate items count from items array
    const calculateItemsCount = (items) => {
        if (!items || !Array.isArray(items)) return 0;
        return items.reduce((total, item) => total + (parseInt(item.quantity) || 0), 0);
    };

    // Safe date formatting
    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        try {
            const date = new Date(dateString);
            return isNaN(date.getTime()) ? 'N/A' : format(date, 'MMM dd, yyyy HH:mm');
        } catch (error) {
            console.error('Date formatting error:', error);
            return 'N/A';
        }
    };

    // Get order timeline based on status
    const getOrderTimeline = (status) => {
        const timeline = [
            { step: 'Order Placed', completed: true, date: 'N/A', icon: ShoppingBagIcon },
            { step: 'Processing', completed: status !== 'pending', date: 'N/A', icon: ClockIcon },
            { step: 'Shipped', completed: status === 'completed', date: 'N/A', icon: TruckIcon },
            { step: 'Delivered', completed: status === 'completed', date: 'N/A', icon: CheckCircleIcon }
        ];
        
        if (status === 'cancelled') {
            timeline.push({ step: 'Cancelled', completed: true, date: 'N/A', isCancelled: true, icon: XCircleIcon });
        }
        
        return timeline;
    };

    // Count orders by status
    const countOrdersByStatus = (status) => {
        return orders?.data ? orders.data.filter(o => o.status === status).length : 0;
    };

    return (
        <AdminLayout>
            <Head title="My Orders" />

            <div className="py-4 sm:py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header Section with Gradient Background */}
                    <div className="bg-gradient-to-r from-indigo-600 to-purple-700 rounded-xl shadow-xl p-4 sm:p-6 mb-6 sm:mb-8 text-white">
                        <h1 className="text-2xl sm:text-3xl font-bold mb-2">My Orders</h1>
                        <p className="text-indigo-100 text-sm sm:text-base">Welcome back, {auth.user.name}. Here you can manage all your product orders.</p>
                    </div>

                    {/* Stats Cards with Enhanced Design */}
                    <div className="grid grid-cols-2 sm:grid-cols-2 lg:grid-cols-4 gap-3 sm:gap-4 mb-6 sm:mb-8">
                        <div className="bg-white overflow-hidden shadow-lg rounded-xl border border-gray-100 hover:shadow-xl transition-shadow duration-300">
                            <div className="p-3 sm:p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0 bg-indigo-500 rounded-lg p-2 sm:p-3">
                                        <ShoppingBagIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                                    </div>
                                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">Total Orders</dt>
                                            <dd className="text-base sm:text-lg font-semibold text-gray-900">{orders?.total || 0}</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-lg rounded-xl border border-gray-100 hover:shadow-xl transition-shadow duration-300">
                            <div className="p-3 sm:p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0 bg-green-500 rounded-lg p-2 sm:p-3">
                                        <CheckCircleIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                                    </div>
                                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">Completed</dt>
                                            <dd className="text-base sm:text-lg font-semibold text-gray-900">{countOrdersByStatus('completed')}</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-lg rounded-xl border border-gray-100 hover:shadow-xl transition-shadow duration-300">
                            <div className="p-3 sm:p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0 bg-yellow-500 rounded-lg p-2 sm:p-3">
                                        <ClockIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                                    </div>
                                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">Pending</dt>
                                            <dd className="text-base sm:text-lg font-semibold text-gray-900">{countOrdersByStatus('pending')}</dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <div className="bg-white overflow-hidden shadow-lg rounded-xl border border-gray-100 hover:shadow-xl transition-shadow duration-300">
                            <div className="p-3 sm:p-5">
                                <div className="flex items-center">
                                    <div className="flex-shrink-0 bg-purple-500 rounded-lg p-2 sm:p-3">
                                        <CurrencyDollarIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                                    </div>
                                    <div className="ml-3 sm:ml-5 w-0 flex-1">
                                        <dl>
                                            <dt className="text-xs sm:text-sm font-medium text-gray-500 truncate">Total Amount</dt>
                                            <dd className="text-base sm:text-lg font-semibold text-gray-900">
                                                GH₵{orders?.data ? orders.data.reduce((sum, order) => sum + parseFloat(order.total_amount || 0), 0).toFixed(2) : '0.00'}
                                            </dd>
                                        </dl>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Search and Filter Section with Enhanced Design */}
                    <div className="bg-white shadow-lg rounded-xl mb-6 p-3 sm:p-4 border border-gray-100">
                        <div className="flex flex-col gap-3">
                            <div className="relative w-full">
                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <MagnifyingGlassIcon className="h-5 w-5 text-gray-400" />
                                </div>
                                <input
                                    type="text"
                                    className="block w-full pl-10 pr-3 py-2 sm:py-3 border border-gray-300 rounded-lg leading-5 bg-white placeholder-gray-500 focus:outline-none focus:placeholder-gray-400 focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 text-sm"
                                    placeholder="Search orders..."
                                    value={searchTerm}
                                    onChange={(e) => setSearchTerm(e.target.value)}
                                />
                            </div>
                            <div className="flex justify-between items-center">
                                <button
                                    onClick={() => setIsFilterExpanded(!isFilterExpanded)}
                                    className="inline-flex items-center px-3 py-2 border border-gray-300 shadow-sm text-sm leading-4 font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                >
                                    <FunnelIcon className="h-4 w-4 mr-2" />
                                    Filters
                                    <ChevronDownIcon className={`ml-2 h-4 w-4 transform transition-transform ${isFilterExpanded ? 'rotate-180' : ''}`} />
                                </button>
                            </div>
                        </div>
                        
                        {isFilterExpanded && (
                            <div className="mt-4 pt-4 border-t border-gray-200">
                                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">Status</label>
                                        <select
                                            className="block w-full px-3 py-2 border border-gray-300 bg-white rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                                            value={filterStatus}
                                            onChange={(e) => setFilterStatus(e.target.value)}
                                        >
                                            <option value="all">All Status</option>
                                            <option value="completed">Completed</option>
                                            <option value="pending">Pending</option>
                                            <option value="cancelled">Cancelled</option>
                                        </select>
                                    </div>
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 mb-1">Date Range</label>
                                        <select
                                            className="block w-full px-3 py-2 border border-gray-300 bg-white rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                                        >
                                            <option value="all">All Time</option>
                                            <option value="today">Today</option>
                                            <option value="week">This Week</option>
                                            <option value="month">This Month</option>
                                            <option value="year">This Year</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        )}
                    </div>

                    {/* Status Tabs - Responsive */}
                    <div className="border-b border-gray-200 mb-6 overflow-x-auto">
                        <nav className="-mb-px flex space-x-2 sm:space-x-8 min-w-max">
                            {['all', 'pending', 'completed', 'cancelled'].map((tab) => (
                                <button
                                    key={tab}
                                    onClick={() => setActiveTab(tab)}
                                    className={`py-2 px-2 sm:px-1 border-b-2 font-medium text-xs sm:text-sm whitespace-nowrap ${
                                        activeTab === tab
                                            ? 'border-indigo-500 text-indigo-600'
                                            : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                    }`}
                                >
                                    {tab.charAt(0).toUpperCase() + tab.slice(1)}
                                    <span className="ml-1 sm:ml-2 bg-gray-100 text-gray-600 py-0.5 px-1.5 sm:px-2 rounded-full text-xs">
                                        {tab === 'all' ? orders?.total || 0 : countOrdersByStatus(tab)}
                                    </span>
                                </button>
                            ))}
                        </nav>
                    </div>

                    {/* Orders List with Enhanced Design - Responsive */}
                    <div className="bg-white shadow-xl rounded-xl overflow-hidden border border-gray-100">
                        {filteredOrders.length > 0 ? (
                            <>
                                {/* Desktop Table View */}
                                <div className="hidden md:block overflow-x-auto">
                                    <table className="min-w-full divide-y divide-gray-200">
                                        <thead className="bg-gray-50">
                                            <tr>
                                                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                    Order Details
                                                </th>
                                                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                    Date
                                                </th>
                                                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                    Items
                                                </th>
                                                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                    Total Amount
                                                </th>
                                                <th scope="col" className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                    Status
                                                </th>
                                                <th scope="col" className="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                    Actions
                                                </th>
                                            </tr>
                                        </thead>
                                        <tbody className="bg-white divide-y divide-gray-200">
                                            {filteredOrders.map((order) => (
                                                <tr key={order.id} className="hover:bg-gray-50 transition-colors">
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <div className="flex items-center">
                                                            <div className="flex-shrink-0 h-10 w-10 bg-indigo-100 rounded-full flex items-center justify-center">
                                                                <ShoppingBagIcon className="h-6 w-6 text-indigo-600" />
                                                            </div>
                                                            <div className="ml-4">
                                                                <div className="text-sm font-medium text-gray-900">Order #{formatOrderId(order.id)}</div>
                                                            </div>
                                                        </div>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <div className="text-sm text-gray-900">{formatDate(order.order_date)}</div>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <div className="text-sm text-gray-900">{order.items_count || calculateItemsCount(order.items)}</div>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <div className="text-sm font-medium text-gray-900">GH₵{formatAmount(order.total_amount)}</div>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(order.status)}`}>
                                                            {getStatusIcon(order.status)}
                                                            <span className="ml-1">{order.status ? order.status.charAt(0).toUpperCase() + order.status.slice(1) : 'Unknown'}</span>
                                                        </span>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                                                        <button 
                                                            onClick={() => viewDetails(order)}
                                                            className="text-indigo-600 hover:text-indigo-900"
                                                        >
                                                            View Details
                                                        </button>
                                                    </td>
                                                </tr>
                                            ))}
                                        </tbody>
                                    </table>
                                </div>

                                {/* Mobile Card View */}
                                <div className="md:hidden">
                                    <ul className="divide-y divide-gray-200">
                                        {filteredOrders.map((order) => (
                                            <li key={order.id} className="p-4 hover:bg-gray-50 transition-colors">
                                                <div className="flex items-start justify-between">
                                                    <div className="flex items-center">
                                                        <div className="flex-shrink-0 h-10 w-10 bg-indigo-100 rounded-full flex items-center justify-center">
                                                            <ShoppingBagIcon className="h-6 w-6 text-indigo-600" />
                                                        </div>
                                                        <div className="ml-3">
                                                            <p className="text-sm font-medium text-gray-900">Order #{formatOrderId(order.id)}</p>
                                                            <p className="text-xs text-gray-500">{formatDate(order.order_date)}</p>
                                                        </div>
                                                    </div>
                                                    <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(order.status)}`}>
                                                        {getStatusIcon(order.status)}
                                                    </span>
                                                </div>
                                                <div className="mt-3 grid grid-cols-2 gap-2 text-sm">
                                                    <div>
                                                        <span className="text-gray-500">Items:</span>
                                                        <span className="ml-1 font-medium">{order.items_count || calculateItemsCount(order.items)}</span>
                                                    </div>
                                                    <div>
                                                        <span className="text-gray-500">Total:</span>
                                                        <span className="ml-1 font-medium">GH₵{formatAmount(order.total_amount)}</span>
                                                    </div>
                                                </div>
                                                <div className="mt-3 flex justify-end">
                                                    <button 
                                                        onClick={() => viewDetails(order)}
                                                        className="text-indigo-600 hover:text-indigo-900 text-sm font-medium"
                                                    >
                                                        View Details
                                                    </button>
                                                </div>
                                            </li>
                                        ))}
                                    </ul>
                                </div>
                            </>
                        ) : (
                            <div className="text-center py-12">
                                <ShoppingBagIcon className="mx-auto h-12 w-12 text-gray-400" />
                                <h3 className="mt-2 text-sm font-medium text-gray-900">No orders found</h3>
                                <p className="mt-1 text-sm text-gray-500">You haven't placed any orders yet.</p>
                                <div className="mt-6">
                                    <Link
                                        href={route('shop')}
                                        className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                    >
                                        Browse Products
                                    </Link>
                                </div>
                            </div>
                        )}
                    </div>

                    {orders?.links && (
                        <Pagination links={orders.links} from={orders.from} to={orders.to} total={orders.total} />
                    )}
                </div>
            </div>

            {/* Enhanced Modal with Modern Design - Responsive */}
            <Modal show={showModal} onClose={closeModal} maxWidth="4xl">
                <div className="bg-white rounded-xl shadow-2xl overflow-hidden max-h-[90vh] overflow-y-auto">
                    {/* Modal Header with Gradient Background */}
                    <div className="bg-gradient-to-r from-indigo-600 to-purple-700 px-4 sm:px-6 py-3 sm:py-4">
                        <div className="flex items-center justify-between">
                            <div className="flex items-center">
                                <div className="flex-shrink-0 bg-white/20 p-2 rounded-lg backdrop-blur-sm">
                                    <ShoppingBagIcon className="h-5 w-5 sm:h-6 sm:w-6 text-white" />
                                </div>
                                <div className="ml-3 sm:ml-4">
                                    <h2 className="text-lg sm:text-xl font-bold text-white">Order Details</h2>
                                    <p className="text-indigo-100 text-xs sm:text-sm">Order #{formatOrderId(selectedOrder?.id)}</p>
                                </div>
                            </div>
                            <button
                                type="button"
                                className="text-white/80 hover:text-white transition-colors p-1 rounded-md hover:bg-white/10"
                                onClick={closeModal}
                            >
                                <XMarkIcon className="h-5 w-5 sm:h-6 sm:w-6" />
                            </button>
                        </div>
                    </div>

                    {/* Modal Body */}
                    <div className="px-4 sm:px-6 py-4 sm:py-5">
                        {selectedOrder ? (
                            <div className="space-y-4 sm:space-y-6">
                                {/* Order Status and Timeline */}
                                <div className="bg-gray-50 rounded-lg p-3 sm:p-4">
                                    <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between mb-3 sm:mb-4">
                                        <h3 className="text-base font-semibold text-gray-900 mb-2 sm:mb-0">Order Status</h3>
                                        <span className={`inline-flex items-center px-3 py-1 rounded-full text-sm font-medium border ${getStatusColor(selectedOrder.status)}`}>
                                            {getStatusIcon(selectedOrder.status)}
                                            <span className="ml-2">{selectedOrder.status ? selectedOrder.status.charAt(0).toUpperCase() + selectedOrder.status.slice(1) : 'Unknown'}</span>
                                        </span>
                                    </div>
                                    
                                    {/* Timeline - Responsive */}
                                    <div className="relative">
                                        <div className="absolute inset-0 flex items-center" aria-hidden="true">
                                            <div className="w-full border-t border-gray-300"></div>
                                        </div>
                                        <div className="relative flex justify-between">
                                            {getOrderTimeline(selectedOrder.status).map((step, index) => (
                                                <div key={index} className="flex flex-col items-center">
                                                    <div className={`flex items-center justify-center w-6 h-6 sm:w-8 sm:h-8 rounded-full border-2 ${
                                                        step.isCancelled 
                                                            ? 'bg-red-100 border-red-300' 
                                                            : step.completed 
                                                                ? 'bg-indigo-600 border-indigo-600' 
                                                                : 'bg-white border-gray-300'
                                                    }`}>
                                                        {step.isCancelled ? (
                                                            <XCircleIcon className="w-3 h-3 sm:w-4 sm:h-4 text-red-600" />
                                                        ) : step.completed ? (
                                                            <step.icon className="w-3 h-3 sm:w-4 sm:h-4 text-white" />
                                                        ) : (
                                                            <div className="w-1.5 h-1.5 sm:w-2 sm:h-2 bg-gray-400 rounded-full"></div>
                                                        )}
                                                    </div>
                                                    <span className={`mt-1 sm:mt-2 text-xs font-medium ${
                                                        step.isCancelled 
                                                            ? 'text-red-600' 
                                                            : step.completed 
                                                                ? 'text-gray-900' 
                                                                : 'text-gray-500'
                                                    }`}>
                                                        {step.step}
                                                    </span>
                                                </div>
                                            ))}
                                        </div>
                                    </div>
                                </div>

                                {/* Order Information Grid - Responsive */}
                                <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                                    <div className="bg-white border border-gray-200 rounded-lg p-3 sm:p-4">
                                        <h3 className="text-sm font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                            <InformationCircleIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-indigo-600" />
                                            Order Information
                                        </h3>
                                        <dl className="space-y-1.5 sm:space-y-2">
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Order ID:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">{formatOrderId(selectedOrder.id)}</dd>
                                            </div>
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Order Date:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">{formatDate(selectedOrder.order_date)}</dd>
                                            </div>
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Payment Method:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">Credit Card</dd>
                                            </div>
                                        </dl>
                                    </div>

                                    <div className="bg-white border border-gray-200 rounded-lg p-3 sm:p-4">
                                        <h3 className="text-sm font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                            <CurrencyDollarIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-indigo-600" />
                                            Price Details
                                        </h3>
                                        <dl className="space-y-1.5 sm:space-y-2">
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Subtotal:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">GH₵{formatAmount(selectedOrder.total_amount)}</dd>
                                            </div>
                                            <div className="flex justify-between">
                                                <dt className="text-xs sm:text-sm text-gray-500">Shipping:</dt>
                                                <dd className="text-xs sm:text-sm font-medium text-gray-900">GH₵0.00</dd>
                                            </div>
                                            <div className="flex justify-between pt-1.5 sm:pt-2 border-t border-gray-200">
                                                <dt className="text-xs sm:text-sm font-semibold text-gray-900">Total:</dt>
                                                <dd className="text-sm sm:text-base font-bold text-indigo-600">GH₵{formatAmount(selectedOrder.total_amount)}</dd>
                                            </div>
                                        </dl>
                                    </div>
                                </div>

                                {/* Customer Information - Responsive */}
                                <div className="bg-white border border-gray-200 rounded-lg p-3 sm:p-4">
                                    <h3 className="text-sm font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                        <UserIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-indigo-600" />
                                        Customer Information
                                    </h3>
                                    <dl className="grid grid-cols-1 sm:grid-cols-2 gap-3 sm:gap-4">
                                        <div>
                                            <dt className="text-xs sm:text-sm text-gray-500">Name:</dt>
                                            <dd className="text-xs sm:text-sm font-medium text-gray-900">{auth.user.name}</dd>
                                        </div>
                                        <div>
                                            <dt className="text-xs sm:text-sm text-gray-500">Email:</dt>
                                            <dd className="text-xs sm:text-sm font-medium text-gray-900">{auth.user.email}</dd>
                                        </div>
                                    </dl>
                                </div>

                                {/* Items List - Responsive */}
                                <div>
                                    <h3 className="text-base font-semibold text-gray-900 mb-2 sm:mb-3 flex items-center">
                                        <ShoppingBagIcon className="h-4 w-4 sm:h-5 sm:w-5 mr-2 text-indigo-600" />
                                        Order Items ({selectedOrder.items_count || calculateItemsCount(selectedOrder.items)})
                                    </h3>
                                    <div className="bg-gray-50 rounded-lg overflow-hidden">
                                        <div className="max-h-48 sm:max-h-64 overflow-y-auto">
                                            {selectedOrder.items && Array.isArray(selectedOrder.items) && selectedOrder.items.length > 0 ? (
                                                <>
                                                    {/* Desktop Table View */}
                                                    <div className="hidden sm:block">
                                                        <table className="min-w-full divide-y divide-gray-200">
                                                            <thead className="bg-gray-100">
                                                                <tr>
                                                                    <th scope="col" className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                                        Product
                                                                    </th>
                                                                    <th scope="col" className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                                        Price
                                                                    </th>
                                                                    <th scope="col" className="px-4 py-2 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                                        Quantity
                                                                    </th>
                                                                    <th scope="col" className="px-4 py-2 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                                        Total
                                                                    </th>
                                                                </tr>
                                                            </thead>
                                                            <tbody className="bg-white divide-y divide-gray-200">
                                                                {selectedOrder.items.map((item, index) => (
                                                                    <tr key={index}>
                                                                        <td className="px-4 py-3 whitespace-nowrap">
                                                                            <div className="flex items-center">
                                                                                <div className="flex-shrink-0 h-10 w-10 bg-gray-200 rounded-lg flex items-center justify-center">
                                                                                    <ShoppingBagIcon className="h-5 w-5 text-gray-400" />
                                                                                </div>
                                                                                <div className="ml-3">
                                                                                    <div className="text-sm font-medium text-gray-900">{item.name || 'Unknown Product'}</div>
                                                                                </div>
                                                                            </div>
                                                                        </td>
                                                                        <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                                                                            GH₵{formatAmount(item.price)}
                                                                        </td>
                                                                        <td className="px-4 py-3 whitespace-nowrap text-sm text-gray-900">
                                                                            {item.quantity || 0}
                                                                        </td>
                                                                        <td className="px-4 py-3 whitespace-nowrap text-sm font-medium text-gray-900 text-right">
                                                                            GH₵{formatAmount((parseFloat(item.price) || 0) * (parseInt(item.quantity) || 0))}
                                                                        </td>
                                                                    </tr>
                                                                ))}
                                                            </tbody>
                                                        </table>
                                                    </div>

                                                    {/* Mobile Card View */}
                                                    <div className="sm:hidden">
                                                        <ul className="divide-y divide-gray-200">
                                                            {selectedOrder.items.map((item, index) => (
                                                                <li key={index} className="p-3">
                                                                    <div className="flex items-center">
                                                                        <div className="flex-shrink-0 h-10 w-10 bg-gray-200 rounded-lg flex items-center justify-center">
                                                                            <ShoppingBagIcon className="h-5 w-5 text-gray-400" />
                                                                        </div>
                                                                        <div className="ml-3 flex-1">
                                                                            <div className="text-sm font-medium text-gray-900">{item.name || 'Unknown Product'}</div>
                                                                            <div className="text-xs text-gray-500">Qty: {item.quantity || 0} × GH₵{formatAmount(item.price)}</div>
                                                                        </div>
                                                                    </div>
                                                                    <div className="mt-2 text-right">
                                                                        <div className="text-sm font-medium text-gray-900">
                                                                            GH₵{formatAmount((parseFloat(item.price) || 0) * (parseInt(item.quantity) || 0))}
                                                                        </div>
                                                                    </div>
                                                                </li>
                                                            ))}
                                                        </ul>
                                                    </div>
                                                </>
                                            ) : (
                                                <div className="text-center py-4">
                                                    <ShoppingBagIcon className="mx-auto h-8 w-8 text-gray-400" />
                                                    <p className="mt-1 text-sm text-gray-500">No items found</p>
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {/* Action Buttons - Responsive */}
                                <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center pt-3 sm:pt-4 border-t border-gray-200">
                                    <div className="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-2 mb-3 sm:mb-0">
                                        <button
                                            type="button"
                                            className="inline-flex items-center justify-center px-3 sm:px-4 py-2 border border-gray-300 rounded-md shadow-sm text-xs sm:text-sm font-medium text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                        >
                                            <ArrowDownTrayIcon className="w-4 h-4 mr-2" />
                                            Download Invoice
                                        </button>
                                        {selectedOrder.status === 'pending' && (
                                            <button
                                                type="button"
                                                className="inline-flex items-center justify-center px-3 sm:px-4 py-2 border border-red-300 rounded-md shadow-sm text-xs sm:text-sm font-medium text-red-700 bg-white hover:bg-red-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500"
                                            >
                                                <TrashIcon className="w-4 h-4 mr-2" />
                                                Cancel Order
                                            </button>
                                        )}
                                    </div>
                                    <button
                                        type="button"
                                        className="w-full sm:w-auto px-4 py-2 bg-indigo-600 text-white rounded-md hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 transition-colors text-sm font-medium"
                                        onClick={closeModal}
                                    >
                                        Close
                                    </button>
                                </div>
                            </div>
                        ) : (
                            <div className="text-center py-8">
                                <InformationCircleIcon className="mx-auto h-10 w-10 text-gray-400" />
                                <p className="mt-2 text-sm text-gray-500">Unable to load order details</p>
                            </div>
                        )}
                    </div>
                </div>
            </Modal>
        </AdminLayout>
    );
}